/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.mobileharness.shared.util.system;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Splitter;
import com.google.common.base.StandardSystemProperty;
import com.google.common.base.Strings;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.flogger.FluentLogger;
import com.google.common.graph.Traverser;
import com.google.devtools.mobileharness.api.model.error.BasicErrorId;
import com.google.devtools.mobileharness.api.model.error.MobileHarnessException;
import com.google.devtools.mobileharness.shared.util.command.Command;
import com.google.devtools.mobileharness.shared.util.command.CommandException;
import com.google.devtools.mobileharness.shared.util.command.CommandExecutor;
import com.google.devtools.mobileharness.shared.util.command.CommandFailureException;
import com.google.devtools.mobileharness.shared.util.command.java.JavaCommandCreator;
import com.google.devtools.mobileharness.shared.util.flags.Flags;
import com.google.devtools.mobileharness.shared.util.path.PathUtil;
import com.google.devtools.mobileharness.shared.util.system.MemoryInfo;
import com.google.devtools.mobileharness.shared.util.system.constant.ExitCode;
import com.google.errorprone.annotations.DoNotCall;
import com.sun.management.OperatingSystemMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.inject.Singleton;

@Singleton
public class SystemUtil {
    private static final FluentLogger logger = FluentLogger.forEnclosingClass();
    public static final String ENV_TEST_UNDECLARED_OUTPUTS_DIR = "TEST_UNDECLARED_OUTPUTS_DIR";
    public static final String ENV_TEST_UNDECLARED_OUTPUTS_ANNOTATIONS_DIR = "TEST_UNDECLARED_OUTPUTS_ANNOTATIONS_DIR";
    public static final String ENV_TEST_DIAGNOSTICS_OUTPUT_DIR = "TEST_DIAGNOSTICS_OUTPUT_DIR";
    public static final String ENV_TEST_INFRASTRUCTURE_FAILURE_FILE = "TEST_INFRASTRUCTURE_FAILURE_FILE";
    public static final String ENV_TEST_COMPONENTS_DIR = "TEST_COMPONENTS_DIR";
    public static final String ENV_XML_OUTPUT_FILE_NAME = "XML_OUTPUT_FILE";
    @VisibleForTesting
    static final String ERROR_MSG_NO_MATCHING_PROCESSES = "No matching processes";
    private static final String ERROR_MSG_NO_MATCHING_SERVICE = "No process to signal.";
    @VisibleForTesting
    static final String ERROR_MSG_NO_PROCESS_FOUND = "no process found";
    private static final Pattern NO_SUCH_PROCESS_PATTERN = Pattern.compile("kill: (.+) No such process");
    private static final Pattern HARDWARE_UUID_PATTERN = Pattern.compile("Hardware UUID: (.+)");
    private static final Pattern PS_PID_PPID_PGID_HEADER_PATTERN = Pattern.compile("^\\s*USER\\s+PID\\s+PPID\\s+PGID\\s+COMMAND\\s*$");
    private static final Pattern PS_PID_PPID_PGID_OUTPUT_PATTERN = Pattern.compile("^\\s*(?<user>\\S+)\\s+(?<pid>\\d+)\\s+(?<ppid>\\d+)\\s+(?<pgid>\\d+)\\s+(?<command>.+)$");
    private static final Pattern UBUNTU_VERSION_PATTERN = Pattern.compile("Description:\\s+(.*)");
    private static final Pattern WHITESPACE_PATTERN = Pattern.compile("\\s+");
    @VisibleForTesting
    static final String GUITAR_CHANGELIST_ENV_VAR = "GUITAR_CHANGELIST";
    private static final Pattern CHANGELIST_PATTERN = Pattern.compile("^\\d+$");
    private static final Pattern PORT_LIST_PATTERN = Pattern.compile("^\\d{1,5}(,\\d{1,5})*$");
    private static final Pattern PORT_RANGE_PATTERN = Pattern.compile("^\\d{1,5}-\\d{1,5}$");
    private static volatile boolean processIsShuttingDown;
    private String osName = System.getProperty("os.name");
    private String archName = System.getProperty("os.arch");
    private final CommandExecutor executor;

    public SystemUtil() {
        this(new CommandExecutor());
    }

    @VisibleForTesting
    SystemUtil(CommandExecutor executor) {
        this.executor = executor;
    }

    public boolean supportDisplay() {
        return !Strings.isNullOrEmpty(System.getenv("DISPLAY"));
    }

    public void exit(com.google.wireless.qa.mobileharness.shared.constant.ExitCode exitCode) {
        this.exit(exitCode.toNewExitCode());
    }

    public void exit(ExitCode exitCode) {
        ((FluentLogger.Api)logger.atInfo()).log("Exit code: %d %s", exitCode.code(), (Object)exitCode.name());
        this.exit(exitCode.code());
    }

    public void exit(int exitCode) {
        System.exit(exitCode);
    }

    public void exit(com.google.wireless.qa.mobileharness.shared.constant.ExitCode exitCode, Throwable e) {
        ((FluentLogger.Api)((FluentLogger.Api)logger.atSevere()).withCause(e)).log("FATAL ERROR");
        this.exit(exitCode);
    }

    public void exit(com.google.wireless.qa.mobileharness.shared.constant.ExitCode exitCode, String severeMsg) {
        ((FluentLogger.Api)logger.atSevere()).log("%s", severeMsg);
        this.exit(exitCode);
    }

    public void exit(com.google.wireless.qa.mobileharness.shared.constant.ExitCode exitCode, Level level, String msg) {
        logger.at(level).log("%s", msg);
        this.exit(exitCode);
    }

    public void exit(com.google.wireless.qa.mobileharness.shared.constant.ExitCode exitCode, String severeMsg, Throwable e) {
        ((FluentLogger.Api)((FluentLogger.Api)logger.atSevere()).withCause(e)).log("%s", severeMsg);
        this.exit(exitCode);
    }

    public String getUser() {
        String userName = System.getenv("USER");
        if (userName != null && !userName.isEmpty()) {
            return userName;
        }
        return StandardSystemProperty.USER_NAME.value();
    }

    public boolean isBlazeTest() {
        return !Strings.isNullOrEmpty(System.getenv(ENV_XML_OUTPUT_FILE_NAME));
    }

    public boolean isRunAsRoot() {
        return this.getUser().equals("root");
    }

    public boolean runAsRoot() {
        return this.isRunAsRoot();
    }

    public String getOsName() {
        return this.osName;
    }

    public String getArchName() {
        return this.archName;
    }

    public boolean isX8664() {
        return this.getArchName().equals("amd64");
    }

    public void setOsNameForTest(String osName) {
        this.osName = osName;
    }

    public String getOsVersion() {
        return StandardSystemProperty.OS_VERSION.value();
    }

    public boolean isOnMac() {
        return this.getOsName().startsWith("Mac OS X");
    }

    public boolean isOnLinux() {
        return this.getOsName().startsWith("Linux");
    }

    public boolean isKvmEnabled() throws MobileHarnessException, InterruptedException {
        if (!this.isOnLinux()) {
            return false;
        }
        try {
            return Integer.parseInt(this.executor.exec(Command.of("grep", "-E", "-c", "'svm|vmx|0xc0f'", "/proc/cpuinfo")).stdoutWithoutTrailingLineTerminator()) > 0 && Files.exists(Paths.get("/dev/kvm", new String[0]), new LinkOption[0]);
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_CHECK_KVM_ERROR, "Failed to check kvm.", e);
        }
    }

    public boolean isQemuInstalled() throws InterruptedException {
        try {
            this.executor.run(Command.of("qemu-system-x86_64", "--version"));
            return true;
        }
        catch (CommandException e) {
            return false;
        }
    }

    @Nullable
    public String getJavaVersion() {
        return StandardSystemProperty.JAVA_VERSION.value();
    }

    public int getJavaFeatureVersion() {
        return Runtime.version().feature();
    }

    public String getJavaBin() {
        String javaBin = this.getEnv("JAVABIN");
        if (javaBin == null) {
            String javaHome = StandardSystemProperty.JAVA_HOME.value();
            if (!Strings.isNullOrEmpty(javaHome)) {
                javaBin = PathUtil.join(System.getProperty("java.home"), "bin", "java");
            } else if (this.isOnMac()) {
                javaBin = "java";
            } else if (this.isOnLinux()) {
                javaBin = "/usr/local/buildtools/java/jdk21/bin/java";
            }
        }
        return javaBin;
    }

    public JavaCommandCreator getJavaCommandCreator() {
        return JavaCommandCreator.of(true, this.getJavaBin());
    }

    @DoNotCall(value="This method may consume GB memory.")
    public final String listOpenFiles() throws MobileHarnessException, InterruptedException {
        try {
            return this.executor.exec(Command.of("lsof")).stdoutWithoutTrailingLineTerminator();
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_LIST_OPEN_FILES_ERROR, "Failed to list open files.", e);
        }
    }

    public String getProcesses() throws MobileHarnessException, InterruptedException {
        String output = "";
        int maxAttempts = 3;
        for (int i = 0; i < maxAttempts; ++i) {
            try {
                output = this.executor.exec(Command.of("ps", "aux")).stdoutWithoutTrailingLineTerminator();
                if (output.split("\n").length <= 1) continue;
                break;
            }
            catch (CommandException e) {
                if (i < maxAttempts - 1) continue;
                throw new MobileHarnessException(BasicErrorId.SYSTEM_LIST_PROCESSES_ERROR, "Failed to list processes.", e);
            }
        }
        return output;
    }

    public String getProcessInfo(int processId) throws MobileHarnessException, InterruptedException {
        try {
            return this.executor.exec(Command.of("ps", "-fp", String.valueOf(processId))).stdoutWithoutTrailingLineTerminator();
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_GET_GIVEN_PROCESS_INFO_ERROR, String.format("Failed to get process info for process id `%s`", processId), e);
        }
    }

    public Set<Integer> getProcessIds(String ... keywords) throws MobileHarnessException, InterruptedException {
        HashSet<Integer> processIds = new HashSet<Integer>();
        String output = this.getProcesses();
        String[] lines = output.split("\n");
        if (lines.length < 1) {
            return processIds;
        }
        String[] words = lines[0].split("\\s+");
        if (words.length < 2 || !words[0].equals("USER") || !words[1].equals("PID")) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_PROCESS_HEADER_NOT_FOUND, "Unexpected command output: header not found\n" + output);
        }
        for (int i = 1; i < lines.length; ++i) {
            String line = lines[i];
            boolean matches = true;
            for (String keyword : keywords) {
                if (line.contains(keyword)) continue;
                matches = false;
                break;
            }
            if (!matches) continue;
            words = line.split("\\s+");
            if (words.length < 2) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_INVALID_PROCESS_INFO_LINE, "\"" + line + "\" is not a valid process info line:\n" + output);
            }
            try {
                processIds.add(Integer.valueOf(words[1]));
                continue;
            }
            catch (NumberFormatException e) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_INVALID_PROCESS_ID, "\"" + words[1] + "\" is not a valid process ID:\n" + output, e);
            }
        }
        return processIds;
    }

    public String getProcessesByKeywords(String ... keywords) throws MobileHarnessException, InterruptedException {
        Object processes = "";
        String output = this.getProcesses();
        List<String> lines = Splitter.on('\n').splitToList(output);
        if (lines.size() < 1) {
            return processes;
        }
        List<String> words = Splitter.onPattern("\\s+").splitToList(lines.get(0));
        if (words.size() < 2 || !words.get(0).equals("USER") || !words.get(1).equals("PID")) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_PROCESS_HEADER_NOT_FOUND, "Unexpected command output: header not found\n" + output);
        }
        for (int i = 1; i < lines.size(); ++i) {
            String line = lines.get(i);
            boolean matches = true;
            for (String keyword : keywords) {
                if (line.contains(keyword)) continue;
                matches = false;
                break;
            }
            if (!matches) continue;
            processes = (String)processes + line + "\n";
        }
        return processes;
    }

    private Set<Integer> getProcessesByPortsString(String ports) throws MobileHarnessException, InterruptedException {
        String[] words;
        int i;
        String output;
        if (!PORT_LIST_PATTERN.matcher(ports).matches() && !PORT_RANGE_PATTERN.matcher(ports).matches()) {
            return new HashSet<Integer>();
        }
        HashSet<Integer> processIds = new HashSet<Integer>();
        try {
            output = this.executor.exec(Command.of("lsof", "-i", ":" + ports)).stdoutWithoutTrailingLineTerminator();
        }
        catch (CommandException e) {
            return processIds;
        }
        String[] lines = output.split("\n");
        if (lines.length < 1) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_NO_PROCESS_FOUND_BY_PORT, "Command output less than 1 line:\n" + output);
        }
        for (i = 0; !(i >= lines.length || (words = lines[i].split("\\s+")).length >= 2 && words[0].equals("COMMAND") && words[1].equals("PID")); ++i) {
        }
        if (i >= lines.length) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_PROCESS_HEADER_NOT_FOUND, "Unexpected command output: header not found:\n" + output);
        }
        ++i;
        while (i < lines.length) {
            String line = lines[i].trim();
            String[] words2 = line.split("\\s+");
            try {
                processIds.add(Integer.valueOf(words2[1]));
            }
            catch (NumberFormatException e) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_INVALID_PROCESS_ID, "\"" + words2[1] + "\" is not a valid process ID:\n" + output, e);
            }
            ++i;
        }
        return processIds;
    }

    public Set<Integer> getProcessesByPortRange(Integer startPort, Integer endPort) throws MobileHarnessException, InterruptedException {
        return this.getProcessesByPortsString(String.format("%d-%d", startPort, endPort));
    }

    public Set<Integer> getProcessesByPorts(List<Integer> ports) throws MobileHarnessException, InterruptedException {
        String portsString = ports.stream().map(String::valueOf).collect(Collectors.joining(","));
        return this.getProcessesByPortsString(portsString);
    }

    public Set<Integer> getProcessesByPort(int port) throws MobileHarnessException, InterruptedException {
        return this.getProcessesByPortsString(String.valueOf(port));
    }

    public void killDescendantAndZombieProcesses(int parentProcessId, KillSignal killSignal) throws MobileHarnessException, InterruptedException {
        this.killDescendantAndZombieProcesses(parentProcessId, killSignal, new HashSet<Integer>(), true);
    }

    private void killDescendantAndZombieProcesses(int parentProcessId, KillSignal killSignal, Set<Integer> ancestorsOfParentProcess, boolean killZombie) throws MobileHarnessException, InterruptedException {
        String output;
        try {
            output = this.executor.exec(Command.of("ps", "xao", "user,pid,ppid,pgid,command")).stdoutWithoutTrailingLineTerminator();
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_LIST_PROCESSES_ERROR, "Failed to list processes.", e);
        }
        String[] lines = output.split("\n");
        if (lines.length < 2) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_INVALID_PROCESS_LIST_ERROR, "Error listing processes:\n" + output);
        }
        Matcher matcher = PS_PID_PPID_PGID_HEADER_PATTERN.matcher(lines[0]);
        if (!matcher.find()) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_UNEXPECTED_PROCESS_HEADER, "Unexpected header:\n" + output);
        }
        ProcessInfo parentProcessInfo = null;
        ArrayListMultimap<Integer, ProcessInfo> processTree = ArrayListMultimap.create();
        HashMap<Integer, Integer> pidToPpid = new HashMap<Integer, Integer>();
        for (String string : lines) {
            Matcher matcher2 = PS_PID_PPID_PGID_OUTPUT_PATTERN.matcher(string);
            if (!matcher2.find()) continue;
            String user = matcher2.group("user");
            int pid = Integer.parseInt(matcher2.group("pid"));
            int ppid = Integer.parseInt(matcher2.group("ppid"));
            int pgid = Integer.parseInt(matcher2.group("pgid"));
            String command = matcher2.group("command").trim();
            ProcessInfo processInfo = new ProcessInfo(user, pid, ppid, pgid, command);
            processTree.put(ppid, processInfo);
            pidToPpid.put(pid, ppid);
            if (pid != parentProcessId) continue;
            parentProcessInfo = processInfo;
        }
        if (parentProcessInfo == null) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_PARENT_PROCESS_NOT_FOUND, String.format("Parent process %d not found. Output:\n%s", parentProcessId, output));
        }
        ((FluentLogger.Api)logger.atInfo()).log("Processes:\n%s", output);
        ((FluentLogger.Api)logger.atInfo()).log("Parent Process for this iteration: %s", parentProcessInfo);
        int ancestorProcessId = (Integer)pidToPpid.get(parentProcessId);
        while (pidToPpid.get(ancestorProcessId) != null && (Integer)pidToPpid.get(ancestorProcessId) != 0) {
            ancestorsOfParentProcess.add(ancestorProcessId);
            ancestorProcessId = (Integer)pidToPpid.get(ancestorProcessId);
        }
        Traverser<ProcessInfo> treeTraverser = Traverser.forTree(n -> processTree.get((Object)n.pid));
        for (ProcessInfo processInfo : treeTraverser.depthFirstPreOrder(parentProcessInfo)) {
            if (processInfo.pid == parentProcessId) continue;
            try {
                this.killProcess(processInfo.pid, killSignal);
                ((FluentLogger.Api)logger.atInfo()).log("Killed %s", processInfo);
            }
            catch (MobileHarnessException e) {
                ((FluentLogger.Api)logger.atWarning()).log("Failed to kill process %s (ignored):\n%s", (Object)processInfo, (Object)e.getMessage());
            }
        }
        if (!killZombie) {
            return;
        }
        for (ProcessInfo processInfo : processTree.get((Object)1)) {
            if (processInfo.pgid != parentProcessInfo.pgid || ancestorsOfParentProcess.contains(processInfo.pid) || processInfo.pid == parentProcessId || processInfo.pid == parentProcessInfo.pgid) continue;
            try {
                this.killDescendantAndZombieProcesses(processInfo.pid, killSignal, ancestorsOfParentProcess, false);
            }
            catch (MobileHarnessException e) {
                ((FluentLogger.Api)logger.atWarning()).log("Failed to kill descendants of process %s (ignored):\n%s", (Object)processInfo, (Object)e.getMessage());
            }
            try {
                this.killProcess(processInfo.pid, killSignal);
                ((FluentLogger.Api)logger.atInfo()).log("Killed %s", processInfo);
            }
            catch (MobileHarnessException e) {
                ((FluentLogger.Api)logger.atWarning()).log("Failed to kill process %s (ignored):\n%s", (Object)processInfo, (Object)e.getMessage());
            }
        }
    }

    public void killProcess(int processId) throws MobileHarnessException, InterruptedException {
        this.killProcess(processId, KillSignal.SIGKILL);
    }

    public void killProcess(int processId, KillSignal killSignal) throws MobileHarnessException, InterruptedException {
        ((FluentLogger.Api)logger.atInfo()).log("Killing process, pid=%s, signal=%s", processId, killSignal.value());
        try {
            this.executor.run(Command.of("kill", "-" + killSignal.value(), String.valueOf(processId)));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_KILL_PROCESS_ERROR, String.format("Failed to kill process [%s] with kill signal value [%s]", processId, killSignal.value()), e);
        }
    }

    public boolean killAllProcesses(String processName) throws InterruptedException, MobileHarnessException {
        return this.killAllProcesses(processName, null);
    }

    public boolean killAllProcesses(String processName, @Nullable KillSignal signal) throws InterruptedException, MobileHarnessException {
        Command command = Command.of("killall");
        if (signal != null) {
            command = command.argsAppended("-" + signal.value());
        }
        command = command.argsAppended(processName);
        try {
            this.executor.run(command);
            return true;
        }
        catch (CommandException e) {
            String exceptionMsg = e.toString();
            if (!exceptionMsg.contains(ERROR_MSG_NO_MATCHING_PROCESSES) && !exceptionMsg.contains(ERROR_MSG_NO_PROCESS_FOUND)) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_KILLALL_PROCESS_ERROR, String.format("Failed to kill the process [%s].", processName), e);
            }
            return false;
        }
    }

    public void addUserToUdevGroup(String user) throws MobileHarnessException, InterruptedException {
        try {
            this.executor.exec(Command.of("adduser", user, "plugdev"));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_ADD_USER_TO_GROUP_ERROR, String.format("Failed to add user `%s` to group `plugdev`.", user), e);
        }
        ((FluentLogger.Api)logger.atInfo()).log("Added user `%s` to group `plugdev`.", user);
    }

    public String getUserGroup(String user) throws MobileHarnessException, InterruptedException {
        String groupList;
        String output;
        try {
            output = this.executor.exec(Command.of("groups", user)).stdoutWithoutTrailingLineTerminator();
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_GET_USER_GROUP_ERROR, String.format("Failed to get groups for user `%s`.", user), e);
        }
        if (this.isOnMac()) {
            groupList = output;
        } else {
            int index = output.indexOf(":");
            if (index < 0) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_FAILED_TO_GET_USER_GROUPS, String.format("Failed to get group of user `%s`: %s", user, output));
            }
            groupList = output.substring(index + 1).trim();
        }
        String[] tokens = groupList.trim().split(" ");
        if (tokens.length < 1) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_NO_GROUPS_FOR_USER, String.format("This is no group of user `%s`, output: %s", user, output));
        }
        return tokens[0].trim();
    }

    public boolean sigkillServiceOnMac(String serviceTargetName, String serviceType) throws InterruptedException, MobileHarnessException {
        if (this.isOnMac()) {
            String[] launchctlListResult;
            String userUid;
            try {
                userUid = this.executor.exec(Command.of("launchctl", "manageruid")).stdoutWithoutTrailingLineTerminator();
            }
            catch (CommandException e) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_GET_USER_GROUP_ERROR, "Failed to get user uid on Mac.", e);
            }
            boolean allCmdSucceeded = true;
            try {
                launchctlListResult = this.executor.exec(Command.of("launchctl", "list")).stdoutWithoutTrailingLineTerminator().split("\n");
            }
            catch (CommandException e) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_MAC_LAUNCHCTL_LIST_ERROR, "Failed to list services on Mac.", e);
            }
            for (String serviceTargetItem : launchctlListResult) {
                String[] items;
                if (!serviceTargetItem.contains(serviceTargetName) || (items = serviceTargetItem.trim().split("\\s+")).length != 3) continue;
                String serviceTarget = serviceTargetItem.trim().split("\\s+")[2];
                try {
                    this.executor.run(Command.of("launchctl", "kill", "SIGKILL", String.format("%s/%s/%s", serviceType, userUid, serviceTarget)));
                    ((FluentLogger.Api)logger.atInfo()).log("Killed %s", serviceTarget);
                }
                catch (CommandException e) {
                    allCmdSucceeded = false;
                    if (e.toString().contains(ERROR_MSG_NO_MATCHING_SERVICE)) continue;
                    ((FluentLogger.Api)logger.atWarning()).log("%s", e);
                }
            }
            return allCmdSucceeded;
        }
        ((FluentLogger.Api)logger.atWarning()).log("It is not on Mac. Failed to sigkill the service.");
        return false;
    }

    public boolean hasProcess(int processId) throws MobileHarnessException, InterruptedException {
        try {
            this.executor.run(Command.of("kill", "-0", String.valueOf(processId)));
            return true;
        }
        catch (CommandException e) {
            if (NO_SUCH_PROCESS_PATTERN.matcher(e.getMessage()).find()) {
                return false;
            }
            throw new MobileHarnessException(BasicErrorId.SYSTEM_KILL_PROCESS_ERROR, String.format("Failed to check if has process id `%s`", processId), e);
        }
    }

    public String getLoginUser() throws MobileHarnessException, InterruptedException {
        String loginUser;
        try {
            loginUser = this.executor.exec(Command.of("logname")).stdoutWithoutTrailingLineTerminator();
        }
        catch (CommandException e) {
            if (e instanceof CommandFailureException && ((CommandFailureException)e).result().stdout().contains("no login name") && this.runAsRoot()) {
                return "root";
            }
            throw new MobileHarnessException(BasicErrorId.SYSTEM_GET_LOGNAME_ERROR, "Failed to get logname.", e);
        }
        if (loginUser.isEmpty()) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_EMPTY_LOGNAME, "Unexpected command output: empty logname.");
        }
        return loginUser;
    }

    public String getUserHome() throws MobileHarnessException, InterruptedException {
        if (this.runAsRoot()) {
            return this.getUserHome(this.getLoginUser());
        }
        String homeEnv = System.getenv("HOME");
        if (homeEnv == null || this.isOnMac()) {
            return System.getProperty("user.home");
        }
        return homeEnv;
    }

    public String getUserHome(String userName) throws MobileHarnessException, InterruptedException {
        if (!this.runAsRoot()) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_ROOT_ACCESS_REQUIRED, "Must run as root.");
        }
        if (this.isOnMac()) {
            String output;
            try {
                output = this.executor.exec(Command.of("dscl", ".", "-read", "/Users/" + userName, "NFSHomeDirectory")).stdoutWithoutTrailingLineTerminator();
            }
            catch (CommandException e) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_MAC_DSCL_CMD_ERROR, "Command dscl failed on Mac.", e);
            }
            String[] array = output.split(" ");
            if (array.length != 2) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_USER_HOME_NOT_FOUND, "Failed to find the user home.");
            }
            return array[1];
        }
        if (this.isOnLinux()) {
            String output;
            try {
                output = this.executor.exec(Command.of("getent", "passwd", userName)).stdoutWithoutTrailingLineTerminator();
            }
            catch (CommandException e) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_GETENT_CMD_ERROR, "Command getent failed.", e);
            }
            String[] array = output.split(":", -1);
            if (array.length != 7) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_USER_HOME_NOT_FOUND, "Failed to find the user home: " + output);
            }
            return array[5];
        }
        throw new MobileHarnessException(BasicErrorId.SYSTEM_NOT_RUN_ON_LINUX_OR_MAC, "Unknown Operation system.");
    }

    @Nullable
    public String getTestUndeclaredOutputsAnnotationsDir() {
        return System.getenv(ENV_TEST_UNDECLARED_OUTPUTS_ANNOTATIONS_DIR);
    }

    @Nullable
    public String getTestUndeclaredOutputDir() {
        return System.getenv(ENV_TEST_UNDECLARED_OUTPUTS_DIR);
    }

    @Nullable
    public String getTestDiagnosticsOutputDir() {
        return System.getenv(ENV_TEST_DIAGNOSTICS_OUTPUT_DIR);
    }

    @Nullable
    public String getTestComponentsOutputDir() {
        return System.getenv(ENV_TEST_COMPONENTS_DIR);
    }

    @Nullable
    public String getEnv(String key) {
        return System.getenv(key);
    }

    public String[] toNonRootCmd(String[] command) throws MobileHarnessException, InterruptedException {
        return this.toNonRootCmd(command, true);
    }

    public String[] toNonRootCmd(String[] command, boolean withEnvVars) throws MobileHarnessException, InterruptedException {
        if (this.isRunAsRoot()) {
            return this.toNonRootCmd(command, this.getLoginUser(), withEnvVars);
        }
        return command;
    }

    public String[] toNonRootCmd(String[] command, String user, boolean withEnvVars) throws MobileHarnessException, InterruptedException {
        if (this.isRunAsRoot()) {
            ArrayList commandList = new ArrayList();
            Collections.addAll(commandList, "sudo");
            if (withEnvVars) {
                Collections.addAll(commandList, "-E");
            }
            Collections.addAll(commandList, "-H");
            Collections.addAll(commandList, "-u", user);
            Collections.addAll(commandList, "--");
            Collections.addAll(commandList, command);
            ((FluentLogger.Api)logger.atInfo()).log("%s", commandList);
            return commandList.toArray(new String[commandList.size()]);
        }
        return command;
    }

    public String[] toNonRootCmdWithoutEnvVar(String[] command) throws MobileHarnessException, InterruptedException {
        return this.toNonRootCmd(command, false);
    }

    public String getJavaCommandPath() {
        return Flags.instance().javaCommandPath.getNonNull();
    }

    public DiskType getDiskType() throws MobileHarnessException, InterruptedException {
        if (this.isOnMac()) {
            String output;
            try {
                output = this.executor.run(Command.of("system_profiler", "SPStorageDataType")).trim();
            }
            catch (CommandException e) {
                throw new MobileHarnessException(BasicErrorId.SYSTEM_GET_MAC_DISK_INFO_ERROR, "Failed to get storage information on Mac.", e);
            }
            if (output.contains("SSD")) {
                return DiskType.SSD;
            }
            if (output.contains("HDD")) {
                return DiskType.HDD;
            }
            ((FluentLogger.Api)logger.atWarning()).log("Failed to determine the disk type according to the output of `system_profiler SPStorageDataType`: %s.", output);
            return DiskType.UNKNOWN;
        }
        throw new MobileHarnessException(BasicErrorId.SYSTEM_GET_DISK_TYPE_NON_MAC_UNIMPLEMENTED, "Unsupported getting disk type on non-macOS machine.");
    }

    public MemoryInfo getMemoryInfo() {
        Runtime runtime = Runtime.getRuntime();
        return MemoryInfo.of(runtime.freeMemory(), runtime.totalMemory(), runtime.maxMemory(), SystemUtil.getMemoryMxBean().map(bean -> bean.getHeapMemoryUsage().getUsed()).orElse(0L), this.getFreeMemory(), this.getTotalMemory());
    }

    public long getTotalMemory() {
        return SystemUtil.getOperatingSystemMxBean().map(OperatingSystemMXBean::getTotalPhysicalMemorySize).orElse(1L);
    }

    public long getFreeMemory() {
        return SystemUtil.getOperatingSystemMxBean().map(OperatingSystemMXBean::getFreePhysicalMemorySize).orElse(0L);
    }

    private static Optional<OperatingSystemMXBean> getOperatingSystemMxBean() {
        try {
            return Optional.of((OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean());
        }
        catch (Error | RuntimeException e) {
            return Optional.empty();
        }
    }

    private static Optional<MemoryMXBean> getMemoryMxBean() {
        try {
            return Optional.of(ManagementFactory.getMemoryMXBean());
        }
        catch (Error | RuntimeException e) {
            return Optional.empty();
        }
    }

    public static boolean isProcessShuttingDown() {
        return processIsShuttingDown;
    }

    public static void setProcessIsShuttingDown() {
        processIsShuttingDown = true;
    }

    public static boolean canRunDockerCommand() {
        try {
            new CommandExecutor().exec(Command.of("docker", "ps"));
            return true;
        }
        catch (CommandException e) {
            return false;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return false;
        }
    }

    public Optional<String> getUbuntuVersion() throws InterruptedException {
        String output = "";
        try {
            output = this.executor.run(Command.of("lsb_release", "-d"));
        }
        catch (MobileHarnessException e) {
            ((FluentLogger.Api)logger.atWarning()).log("Failed to get Ubuntu version: %s", e.getMessage());
            return Optional.empty();
        }
        Matcher matcher = UBUNTU_VERSION_PATTERN.matcher(output);
        if (matcher.find()) {
            return Optional.of(matcher.group(1));
        }
        ((FluentLogger.Api)logger.atWarning()).log("Failed to get Ubuntu version. The output is %s", output);
        return Optional.empty();
    }

    public Optional<String> getProcessWorkingDirectory(long processId) throws InterruptedException {
        if (!this.isOnLinux()) {
            return Optional.empty();
        }
        String output = null;
        try {
            output = this.executor.run(Command.of("pwdx", String.valueOf(processId)));
            return Optional.of(Iterables.get(Splitter.on(' ').split(output), 1));
        }
        catch (CommandException | IndexOutOfBoundsException e) {
            ((FluentLogger.Api)logger.atWarning()).log("Failed to get process working directory for process id %d with output [%s]", processId, (Object)String.valueOf(output));
            return Optional.empty();
        }
    }

    public Optional<String> getRootDiskSpace() throws InterruptedException {
        String output = "";
        try {
            output = this.executor.run(Command.of("df", "-h", "/"));
            return Optional.of(Iterables.get(Splitter.on(WHITESPACE_PATTERN).split(Iterables.get(Splitter.on('\n').split(output), 1)), 1));
        }
        catch (CommandException | IndexOutOfBoundsException e) {
            ((FluentLogger.Api)logger.atWarning()).log("Failed to get root disk space with error %s and output %s", (Object)e.getMessage(), (Object)output);
            return Optional.empty();
        }
    }

    public static enum KillSignal {
        SIGINT(2),
        SIGKILL(9),
        SIGTERM(15),
        SIGTSTP(20);

        private final int value;

        private KillSignal(int value) {
            this.value = value;
        }

        public int value() {
            return this.value;
        }
    }

    private static class ProcessInfo {
        public final String user;
        public final int pid;
        public final int ppid;
        public final int pgid;
        public final String command;

        public ProcessInfo(String user, int pid, int ppid, int pgid, String command) {
            this.user = user;
            this.pid = pid;
            this.ppid = ppid;
            this.pgid = pgid;
            this.command = command;
        }

        public String toString() {
            return String.format("Process: user=%s, pid=%d, ppid=%d, pgid=%d, command=%s", this.user, this.pid, this.ppid, this.pgid, this.command);
        }
    }

    public static enum DiskType {
        UNKNOWN(0),
        HDD(1),
        SSD(2);

        private final int value;

        private DiskType(int value) {
            this.value = value;
        }

        public int value() {
            return this.value;
        }
    }
}

