# Copyright 2021 The Android Open Source Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Utility functions for sensor_fusion hardware rig."""


import logging
import struct
import time
import gen2_rig_controller_utils
import sensor_fusion_utils

# Constants for Arduino
ARDUINO_BRIGHTNESS_MAX = 255
ARDUINO_BRIGHTNESS_MIN = 0
ARDUINO_LIGHT_START_BYTE = 254
LIGHT_ON = 'ON'
LIGHT_OFF = 'OFF'


def _toggle_screen_state(device, desired_state):
  """Trigger device power key to toggle screen."""
  current_state = 'ON'
  if desired_state == 'ON':
    current_state = 'OFF'
  output = device.adb.shell('dumpsys display | grep mScreenState=')
  output_val = str(output.decode('utf-8')).strip()
  if current_state in output_val:
    device.adb.shell(['input', 'keyevent', 'KEYCODE_POWER'])


def turn_off_device_screen(device):
  """Turn off a device screen via power key if screen is on."""
  _toggle_screen_state(device, 'OFF')


def turn_on_device_screen(device):
  """Turn on a device screen via power key if screen is on."""
  _toggle_screen_state(device, 'ON')


def set_light_brightness(ch, brightness, serial_port, delay=0):
  """Turn on light to specified brightness.

  Args:
    ch: str; light to turn on in ARDUINO_VALID_CH
    brightness: int value of brightness between 0 and 255.
    serial_port: object; serial port
    delay: int; time in seconds
  """
  if brightness < ARDUINO_BRIGHTNESS_MIN:
    logging.debug('Brightness must be >= %d.', ARDUINO_BRIGHTNESS_MIN)
    brightness = ARDUINO_BRIGHTNESS_MIN
  elif brightness > ARDUINO_BRIGHTNESS_MAX:
    logging.debug('Brightness must be <= %d.', ARDUINO_BRIGHTNESS_MAX)
    brightness = ARDUINO_BRIGHTNESS_MAX

  cmd = [struct.pack('B', i) for i in [
      ARDUINO_LIGHT_START_BYTE, int(ch), brightness]]
  logging.debug('cmd: %s', cmd)
  sensor_fusion_utils.arduino_send_cmd(serial_port, cmd)
  time.sleep(delay)


def lighting_control(lighting_cntl, lighting_ch, use_gen2=False):
  """Establish communication with lighting controller.

  lighting_ch is hard wired and must be determined from physical setup.

  First initialize the port and send a test string defined by ARDUINO_TEST_CMD
  to establish communications.

  Args:
    lighting_cntl: str to identify 'arduino' controller.
    lighting_ch: str to identify lighting channel number.
    use_gen2: optional bool to identify if gen2 lighting controller is used.
  Returns:
    serial port pointer
  """
  if use_gen2:
    logging.debug(
        'Gen2 lighting controller: %s, ch: %s', lighting_cntl, lighting_ch)
    lights_port = gen2_rig_controller_utils.find_serial_port(lighting_cntl)
    sensor_fusion_utils.establish_serial_comm(lights_port)
    return lights_port
  elif lighting_cntl.lower() == 'arduino':
    logging.debug(
        'Arduino lighting Controller: %s, ch: %s', lighting_cntl, lighting_ch)
    # identify port
    arduino_serial_port = sensor_fusion_utils.serial_port_def('arduino')

    # send test cmd to Arduino until cmd returns properly
    sensor_fusion_utils.establish_serial_comm(arduino_serial_port)

    # return serial port
    return arduino_serial_port
  else:
    logging.debug('No lighting control: need to control lights manually.')
    return None


def set_lighting_state(arduino_serial_port, lighting_ch, state, use_gen2=False):
  """Turn lights ON in test rig.

  Args:
    arduino_serial_port: serial port object
    lighting_ch: str for lighting channel
    state: str 'ON/OFF'
    use_gen2: optional bool to identify if gen2 lighting controller is used.
  """
  if use_gen2:
    logging.debug(
        'Setting lighting state to %s using Gen2 lighting controller.', state)
    gen2_rig_controller_utils.set_lighting_state(
        arduino_serial_port, int(lighting_ch), state)
    return

  if state == LIGHT_ON:
    level = 255
  elif state == LIGHT_OFF:
    level = 0
  else:
    raise ValueError(f'Lighting state not defined correctly: {state}')

  if arduino_serial_port:
    set_light_brightness(lighting_ch, level, arduino_serial_port, delay=1)
  else:
    print(f'Turn {state} lights in rig and hit <ENTER> to continue.')
    input('')

