/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.mobileharness.shared.util.file.local;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.flogger.FluentLogger;
import com.google.devtools.deviceinfra.shared.util.file.remote.constant.RemoteFileType;
import com.google.devtools.mobileharness.api.model.error.BasicErrorId;
import com.google.devtools.mobileharness.api.model.error.MobileHarnessException;
import com.google.devtools.mobileharness.shared.util.command.Command;
import com.google.devtools.mobileharness.shared.util.command.CommandException;
import com.google.devtools.mobileharness.shared.util.command.CommandExecutor;
import com.google.devtools.mobileharness.shared.util.command.Timeout;
import com.google.devtools.mobileharness.shared.util.path.PathUtil;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import java.util.zip.ZipFile;
import javax.annotation.Nullable;
import javax.inject.Singleton;

@Singleton
public class LocalFileUtil {
    private static final FluentLogger logger = FluentLogger.forEnclosingClass();
    public static final FileAttribute<Set<PosixFilePermission>> FULL_ACCESS = PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rwxrwxrwx"));
    private static final int BUFFER_SIZE = 32768;
    private static final int LINK_ATTEMPTS = 100;
    private static final Duration SLOW_CMD_TIMEOUT = Duration.ofMinutes(10L);
    private static final Pattern SPACE_CHARS = Pattern.compile("\\s+");
    private final CommandExecutor cmdExecutor;

    public LocalFileUtil() {
        this(new CommandExecutor());
    }

    @VisibleForTesting
    public LocalFileUtil(CommandExecutor cmdExecutor) {
        this.cmdExecutor = cmdExecutor;
    }

    public void appendToFile(Path srcFile, Path dstFile) throws MobileHarnessException {
        this.checkFile(srcFile);
        try (SeekableByteChannel out = Files.newByteChannel(dstFile, EnumSet.of(StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.APPEND), new FileAttribute[0]);){
            ByteBuffer buffer = ByteBuffer.allocate(32768);
            try (SeekableByteChannel in = Files.newByteChannel(srcFile, new OpenOption[0]);){
                while (in.read(buffer) >= 0) {
                    in.read(buffer);
                    buffer.flip();
                    out.write(buffer);
                    buffer.clear();
                }
            }
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_APPEND_ERROR, String.format("Failed to append file %s to file %s", srcFile, dstFile), e);
        }
    }

    public String changeFileOrDirGroup(String fileOrDirPath, String group) throws MobileHarnessException, InterruptedException {
        this.checkFileOrDir(fileOrDirPath);
        try {
            return this.cmdExecutor.run(Command.of("chgrp", "-R", group, fileOrDirPath));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_CHANGE_GROUP_ERROR, String.format("Failed to change the file/dir %s to group %s", fileOrDirPath, group), e);
        }
    }

    public String changeFileOrDirOwner(String fileOrDirPath, String user) throws MobileHarnessException, InterruptedException {
        this.checkFileOrDir(fileOrDirPath);
        try {
            return this.cmdExecutor.run(Command.of("chown", "-R", user, fileOrDirPath));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_CHANGE_OWNER_ERROR, String.format("Failed to change the file/dir %s to owner %s", fileOrDirPath, user), e);
        }
    }

    public File checkDir(String dirPath) throws MobileHarnessException {
        File dir = this.checkFileOrDir(dirPath);
        if (dir.isFile()) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_IS_FILE, String.valueOf(dir) + " is not a directory");
        }
        return dir;
    }

    public Path checkDir(Path dir) throws MobileHarnessException {
        if (!Files.isDirectory(dir = this.checkFileOrDir(dir), new LinkOption[0])) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_IS_FILE, String.valueOf(dir) + " is not a directory");
        }
        return dir;
    }

    public File checkFile(String filePath) throws MobileHarnessException {
        File file = this.checkFileOrDir(filePath);
        if (file.isDirectory()) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_IS_DIR, String.valueOf(file) + " is not a file");
        }
        return file;
    }

    public Path checkFile(Path file) throws MobileHarnessException {
        if (Files.isDirectory(file = this.checkFileOrDir(file), new LinkOption[0])) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_IS_DIR, String.valueOf(file) + " is not a file");
        }
        return file;
    }

    public File checkFileOrDir(String fileOrDirPath) throws MobileHarnessException {
        File fileOrDir = this.getFileOrDir(fileOrDirPath);
        if (!fileOrDir.exists()) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_NOT_FOUND, "Can not find file or directory: " + fileOrDirPath);
        }
        return fileOrDir;
    }

    @CanIgnoreReturnValue
    public Path checkFileOrDir(Path fileOrDir) throws MobileHarnessException {
        if (!Files.exists(fileOrDir, new LinkOption[0])) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_NOT_FOUND, "Can not find file or directory: " + String.valueOf(fileOrDir));
        }
        return fileOrDir;
    }

    public void clearUnopenedFiles(Path dirPath) throws MobileHarnessException, InterruptedException {
        this.clearUnopenedFiles(dirPath, null);
    }

    public void clearUnopenedFiles(Path dirPath, @Nullable Duration expiration) throws MobileHarnessException, InterruptedException {
        this.checkDir(dirPath);
        Command openFilesCommand = Command.of("lsof", "-w", "-Fn", "+D", dirPath.toString()).timeout(Duration.ofSeconds(30L)).successExitCodes(0, 1);
        ImmutableSet.Builder allOpenedFilesOrDirsBuilder = ImmutableSet.builder();
        try {
            String cmdOutput = this.cmdExecutor.exec(openFilesCommand).stdout();
            Iterable<String> allOpenedFilesOrDirsArray = Splitter.on("\n").split(cmdOutput);
            for (String line : allOpenedFilesOrDirsArray) {
                if (!line.startsWith("n")) continue;
                allOpenedFilesOrDirsBuilder.add(line.substring(1));
            }
        }
        catch (CommandException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atWarning()).withCause(e)).log("Failed to list open files or directories under %s", dirPath);
        }
        ImmutableCollection allOpenedFilesOrDirs = allOpenedFilesOrDirsBuilder.build();
        ((FluentLogger.Api)logger.atInfo()).log("Opened files or dirs: %s", allOpenedFilesOrDirs);
        Command listRunningCommand = Command.of("ps", "xao", "command").timeout(Duration.ofSeconds(30L));
        String runningCommands = "";
        try {
            String commandOutput = this.cmdExecutor.exec(listRunningCommand).stdout();
            ImmutableList.Builder runningCommandsBuilder = ImmutableList.builder();
            Iterable<String> runningCommandsArray = Splitter.on("\n").split(commandOutput);
            Iterator<String> iterator2 = runningCommandsArray.iterator();
            while (iterator2.hasNext()) {
                String line = iterator2.next();
                if (!line.contains(dirPath.toString())) continue;
                runningCommandsBuilder.add(line);
            }
            runningCommands = runningCommandsBuilder.build().toString();
            ((FluentLogger.Api)logger.atInfo()).log("Running Commands: %s with file under dir: %s", (Object)runningCommands, (Object)dirPath);
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.SYSTEM_LIST_PROCESSES_ERROR, String.format("Failed to list processes used files under %s", dirPath), e);
        }
        List<Path> allFiles = this.listFilePaths(dirPath, true);
        ImmutableList.Builder removedFilesBuilder = ImmutableList.builder();
        for (Path filePath : allFiles) {
            if (allOpenedFilesOrDirs.contains(filePath.toString())) {
                ((FluentLogger.Api)logger.atInfo()).log("Skip opened file or dir: %s", filePath);
                continue;
            }
            if (runningCommands.contains(filePath.toString())) {
                ((FluentLogger.Api)logger.atInfo()).log("Skip file or dir listed in running commands: %s", filePath);
                continue;
            }
            if (expiration != null && this.isFileChanged(filePath, expiration)) continue;
            try {
                this.removeFileOrDir(filePath);
                removedFilesBuilder.add(filePath);
            }
            catch (MobileHarnessException e) {
                ((FluentLogger.Api)((FluentLogger.Api)logger.atWarning()).withCause(e)).log("Failed to remove unopened file or dir %s.", filePath);
            }
        }
        ImmutableCollection removedFiles = removedFilesBuilder.build();
        if (removedFiles.isEmpty()) {
            ((FluentLogger.Api)logger.atInfo()).log("No matched unopened files removed under %s", dirPath);
        } else {
            ((FluentLogger.Api)logger.atInfo()).log("Removed unopened files: %s", removedFiles);
        }
    }

    public boolean isFileChanged(Path path, Duration duration) throws MobileHarnessException {
        Instant lastModifiedTime = this.getFileLastModifiedTime(path);
        Instant now = Instant.now();
        return lastModifiedTime.plus(duration).isAfter(now);
    }

    public void copyFileOrDir(final String srcFileOrDirPath, String desFileOrDirPath) throws MobileHarnessException, InterruptedException {
        ((FluentLogger.Api)logger.atInfo()).log("Copy file or dir from %s to %s", (Object)srcFileOrDirPath, (Object)desFileOrDirPath);
        if (Files.isDirectory(Paths.get(desFileOrDirPath, new String[0]), new LinkOption[0])) {
            desFileOrDirPath = PathUtil.join(desFileOrDirPath, PathUtil.basename(srcFileOrDirPath));
        }
        try {
            final String finalDesFileOrDirPath = desFileOrDirPath;
            Files.walkFileTree(Paths.get(srcFileOrDirPath, new String[0]), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    Path targetFile = Paths.get(finalDesFileOrDirPath, dir.toString().substring(srcFileOrDirPath.length()));
                    if (!Files.exists(targetFile, new LinkOption[0])) {
                        Files.createDirectory(targetFile, new FileAttribute[0]);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Path targetFile = Path.of(finalDesFileOrDirPath, file.toString().substring(srcFileOrDirPath.length()));
                    try {
                        Files.copy(file, targetFile, LinkOption.NOFOLLOW_LINKS, StandardCopyOption.REPLACE_EXISTING);
                    }
                    catch (NoSuchFileException noSuchFileException) {
                        // empty catch block
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    if (exc instanceof NoSuchFileException) {
                        return FileVisitResult.CONTINUE;
                    }
                    throw exc;
                }
            });
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_COPY_ERROR, "Failed to copy file or dir", e);
        }
    }

    public void copyFileOrDir(Path srcFileOrDir, Path desFileOrDir) throws MobileHarnessException, InterruptedException {
        this.copyFileOrDir(srcFileOrDir.toAbsolutePath().toString(), desFileOrDir.toAbsolutePath().toString());
    }

    public void copyFileOrDirWithOverridingCopyOptions(Path srcFileOrDir, Path desFileOrDir, List<String> overridingCopyOptions) throws MobileHarnessException, InterruptedException {
        try {
            this.cmdExecutor.exec(Command.of("cp", Stream.concat(overridingCopyOptions.stream(), Stream.of(srcFileOrDir.toAbsolutePath().toString(), desFileOrDir.toAbsolutePath().toString())).collect(ImmutableList.toImmutableList())));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_COPY_ERROR, "Failed to copy file or dir", e);
        }
    }

    public String createTempDir(String parentDirPath) throws MobileHarnessException {
        String tempDirectoryPath = PathUtil.join(parentDirPath, UUID.randomUUID().toString());
        try {
            this.prepareDir(tempDirectoryPath, FULL_ACCESS);
        }
        catch (MobileHarnessException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_CREATE_TMP_ERROR, "Failed to create temp dir under " + parentDirPath, e);
        }
        return tempDirectoryPath;
    }

    public Path createTempDir(Path parentDirPath, String fileNamePrefix) throws MobileHarnessException {
        try {
            return Files.createTempDirectory(parentDirPath, fileNamePrefix, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_CREATE_TMP_ERROR, "Failed to create temp dir under " + String.valueOf(parentDirPath), e);
        }
    }

    public Path createTempFile(Path parentDirPath, String fileNamePrefix, String fileNameSuffix) throws MobileHarnessException {
        try {
            return Files.createTempFile(parentDirPath, fileNamePrefix, fileNameSuffix, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_CREATE_TMP_ERROR, "Failed to create temp file under " + String.valueOf(parentDirPath), e);
        }
    }

    public String createTempFile(String parentDirPath, String fileNamePrefix, String fileNameSuffix) throws MobileHarnessException {
        return this.createTempFile(Paths.get(parentDirPath, new String[0]), fileNamePrefix, fileNameSuffix).toString();
    }

    public Instant getFileLastModifiedTime(String filePath) throws MobileHarnessException {
        return this.getFileLastModifiedTime(Paths.get(filePath, new String[0]));
    }

    public Instant getFileLastModifiedTime(Path file) throws MobileHarnessException {
        try {
            return Instant.ofEpochMilli(Files.getLastModifiedTime(file, new LinkOption[0]).toMillis());
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_GET_MODIFIED_TIME_ERROR, String.format("Failed to get the last modified time of file %s", file), e);
        }
    }

    public File getFileOrDir(String fileOrDirPath) {
        return new File(fileOrDirPath);
    }

    public String getFileOrDirHumanReadableSize(String fileOrDirPath) throws MobileHarnessException, InterruptedException {
        String output;
        this.checkFileOrDir(fileOrDirPath);
        try {
            output = this.cmdExecutor.run(Command.of("du", "-sh", fileOrDirPath).timeout(Duration.ofMinutes(10L))).trim();
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_GET_SIZE_ERROR, "Failed to check the size of file/dir " + fileOrDirPath, e);
        }
        List<String> words = Splitter.on(SPACE_CHARS).splitToList(output);
        if (words.size() != 2) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_GET_SIZE_ERROR, String.format("Failed to parse the size of file/dir %s from:%n%s", fileOrDirPath, output));
        }
        return words.get(0);
    }

    public String getFileOrDirName(String fileOrDirPath) {
        return this.getFileOrDir(fileOrDirPath).getName();
    }

    public String getFileOrDirNameWithoutExtension(String fileOrDirPath) {
        String fileOrDirName = this.getFileOrDirName(fileOrDirPath);
        int dotIndex = fileOrDirName.lastIndexOf(46);
        return dotIndex == -1 ? fileOrDirName : fileOrDirName.substring(0, dotIndex);
    }

    public Path getFileOrDirRealPath(Path fileOrDir) throws MobileHarnessException {
        try {
            return fileOrDir.toRealPath(new LinkOption[0]);
        }
        catch (IOException | SecurityException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_GET_REAL_PATH_ERROR, "Failed to get real path of " + String.valueOf(fileOrDir), e);
        }
    }

    public long getFileOrDirSize(String fileOrDirPath) throws MobileHarnessException, InterruptedException {
        String output;
        this.checkFileOrDir(fileOrDirPath);
        try {
            output = this.cmdExecutor.run(Command.of("du", "-s", fileOrDirPath).timeout(SLOW_CMD_TIMEOUT)).trim();
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_GET_SIZE_ERROR, "Failed to check the size of file/dir " + fileOrDirPath, e);
        }
        List<String> words = Splitter.on(SPACE_CHARS).splitToList(output);
        if (words.size() < 2) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_PARSE_SIZE_ERROR, String.format("Failed to parse the size of file/dir %s from:%n%s", fileOrDirPath, output));
        }
        try {
            return Long.parseLong(words.get(0)) << 10;
        }
        catch (NumberFormatException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_PARSE_SIZE_ERROR, String.format("Failed to parse the size of file/dir %s from:%n%s", fileOrDirPath, output), e);
        }
    }

    public String getFileOwner(Path file) throws MobileHarnessException {
        try {
            return Files.getOwner(file, new LinkOption[0]).getName();
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_GET_OWNER_ERROR, "Failed to get owner of file " + String.valueOf(file), e);
        }
    }

    public Set<PosixFilePermission> getFilePermission(Path file) throws MobileHarnessException {
        try {
            return Files.getPosixFilePermissions(file, new LinkOption[0]);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_GET_PERMISSION_ERROR, "Failed to get permission of file " + String.valueOf(file), e);
        }
    }

    public String getFilePermissionString(Path file) throws MobileHarnessException {
        return PosixFilePermissions.toString(this.getFilePermission(file));
    }

    public long getFileSize(Path file) throws MobileHarnessException {
        try {
            return Files.size(file);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_GET_SIZE_ERROR, "Failed to get size of file: " + String.valueOf(file), e);
        }
    }

    public long getFileSize(String filePath) throws MobileHarnessException {
        return this.getFileSize(Paths.get(filePath, new String[0]));
    }

    public String getParentDirPath(String fileOrDirPath) {
        File parentDir = this.getFileOrDir(fileOrDirPath).getParentFile();
        return parentDir == null ? "" : parentDir.getAbsolutePath();
    }

    public void grantFileOrDirFullAccess(String fileOrDirPath) throws MobileHarnessException {
        if (this.isFullyAccessible(Paths.get(fileOrDirPath, new String[0]))) {
            return;
        }
        File fileOrDir = this.checkFileOrDir(fileOrDirPath);
        ArrayList<String> errors = new ArrayList<String>();
        if (!fileOrDir.setReadable(true, false)) {
            errors.add("read");
        }
        if (!fileOrDir.setWritable(true, false)) {
            errors.add("write");
        }
        if (!fileOrDir.setExecutable(true, false)) {
            errors.add("execution");
        }
        if (errors.isEmpty()) {
            return;
        }
        String message = String.format("Fail to grant %s access to %s", errors, fileOrDir);
        throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_GRANT_PERMISSION_ERROR, message);
    }

    public void grantFileOrDirFullAccess(Path fileOrDir) throws MobileHarnessException {
        this.grantFileOrDirFullAccess(fileOrDir.toString());
    }

    public void grantFileOrDirFullAccessRecursively(String fileOrDirPath) throws MobileHarnessException, InterruptedException {
        if (this.areAllFilesFullAccessible(Paths.get(fileOrDirPath, new String[0]))) {
            return;
        }
        File fileOrDir = this.checkFileOrDir(fileOrDirPath);
        try {
            this.cmdExecutor.exec(Command.of("chmod", "-R", "777", fileOrDirPath));
        }
        catch (MobileHarnessException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_GRANT_PERMISSION_RECURSIVELY_ERROR, "Fail to grant full access recursively to " + String.valueOf(fileOrDir), e);
        }
    }

    public void grantFileOrDirFullAccessRecursively(Path fileOrDirPath) throws MobileHarnessException, InterruptedException {
        this.grantFileOrDirFullAccessRecursively(fileOrDirPath.toString());
    }

    public boolean isDirExist(String dirPath) {
        try {
            this.checkDir(dirPath);
            return true;
        }
        catch (MobileHarnessException e) {
            return false;
        }
    }

    public boolean isDirExist(Path dir) {
        try {
            this.checkDir(dir);
            return true;
        }
        catch (MobileHarnessException e) {
            return false;
        }
    }

    public boolean isFileExist(String filePath) {
        try {
            this.checkFile(filePath);
            return true;
        }
        catch (MobileHarnessException e) {
            return false;
        }
    }

    public boolean isFileExist(Path file) {
        try {
            this.checkFile(file);
            return true;
        }
        catch (MobileHarnessException e) {
            return false;
        }
    }

    public boolean isFileOrDirExist(String fileOrDirPath) {
        try {
            this.checkFileOrDir(fileOrDirPath);
            return true;
        }
        catch (MobileHarnessException e) {
            return false;
        }
    }

    public boolean isFileOrDirExist(Path fileOrDir) {
        try {
            this.checkFileOrDir(fileOrDir);
            return true;
        }
        catch (MobileHarnessException e) {
            return false;
        }
    }

    public boolean isZipFileValid(String filePath) {
        try {
            ZipFile zipFile = new ZipFile(filePath);
            ((FluentLogger.Api)logger.atInfo()).log("Zip file %s is valid.", zipFile.getName());
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public boolean isFileExistInPath(String filePath) {
        if (this.isFileExist(filePath)) {
            return true;
        }
        if (filePath.contains(File.separator)) {
            return false;
        }
        return Arrays.stream(System.getenv("PATH").split(Pattern.quote(File.pathSeparator))).map(x$0 -> Path.of(x$0, new String[0])).anyMatch(path -> this.isFileExist(path.resolve(filePath)));
    }

    public boolean isLocalFileOrDir(String fileOrDirPath) {
        for (RemoteFileType fileType : RemoteFileType.values()) {
            if (!fileOrDirPath.startsWith(fileType.prefix())) continue;
            return false;
        }
        return true;
    }

    public boolean isSymbolicLink(Path file) {
        try {
            return Files.isSymbolicLink(file);
        }
        catch (SecurityException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atWarning()).withCause(e)).log("Failed to check whether file %s is a symbolic link, return false.", file);
            return false;
        }
    }

    public boolean isHardLink(Path file) {
        try {
            return (Integer)Files.getAttribute(file, "unix:nlink", new LinkOption[0]) >= 2;
        }
        catch (IOException | SecurityException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atWarning()).withCause(e)).log("Failed to check whether file %s is a hard link, return false.", file);
            return false;
        }
    }

    public Path linkDir(String linkBaseDirPath, String linkPrefixName, String targetDirPath) throws MobileHarnessException {
        IOException lastException = null;
        for (int counter = 0; counter < 100; ++counter) {
            File linkDir = new File(linkBaseDirPath, linkPrefixName + counter);
            if (linkDir.exists()) continue;
            try {
                return Files.createSymbolicLink(linkDir.toPath(), Paths.get(targetDirPath, new String[0]), new FileAttribute[0]);
            }
            catch (IOException e) {
                if (e instanceof FileAlreadyExistsException && counter < 99) {
                    lastException = e;
                    continue;
                }
                throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_LINK_ERROR, String.format("Failed to create symbolic link. link: %s, target: %s ", linkDir, targetDirPath), e);
            }
        }
        throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_LINK_ERROR_WITH_RETRY, "Failed to create directory within 100 attempts (tried " + linkPrefixName + "0 to " + linkPrefixName + "99)", lastException);
    }

    public void linkFileOrDir(String targetFileOrDirPath, String linkName) throws MobileHarnessException, InterruptedException {
        this.checkFileOrDir(targetFileOrDirPath);
        try {
            this.cmdExecutor.exec(Command.of("ln", "-sf", targetFileOrDirPath, linkName));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_LINK_ERROR, "Failed to create symbolic link for " + targetFileOrDirPath, e);
        }
    }

    public void hardLinkFile(String targetFilePath, String linkName) throws MobileHarnessException, InterruptedException {
        this.checkFile(targetFilePath);
        try {
            this.cmdExecutor.exec(Command.of("ln", "-f", targetFilePath, linkName));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_CREATE_HARD_LINK_ERROR, "Failed to create hard link for " + targetFilePath, e);
        }
    }

    public Set<String> listAllFilesBeenLinked(String dirPath) throws MobileHarnessException, InterruptedException {
        try {
            String[] files = this.cmdExecutor.run(Command.of("find", dirPath, "-type", "l", "-exec", "readlink", "{}", ";")).split("\n");
            return new HashSet<String>(Arrays.asList(files));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_LIST_LINKS_ERROR, "Failed to list the linked files under " + dirPath, e);
        }
    }

    public File[] listDirs(String dirPath) throws MobileHarnessException {
        return this.listFilesOrDirs(dirPath, File::isDirectory);
    }

    public File[] listDirs(String dirPath, @Nullable FileFilter dirFilter) throws MobileHarnessException {
        return this.listFilesOrDirs(dirPath, (File pathname) -> pathname.isDirectory() && (dirFilter == null || dirFilter.accept(pathname)));
    }

    public List<Path> listDirs(Path dir) throws MobileHarnessException {
        return this.listFilesOrDirs(dir, (Path x$0) -> Files.isDirectory(x$0, new LinkOption[0]));
    }

    public List<String> listDirs(String dirPath, int depth, @Nullable FileFilter fileFilter) throws MobileHarnessException {
        return this.listDirs(dirPath, depth, false, fileFilter);
    }

    public List<String> listDirs(String dirPath, int depth, boolean recursively, @Nullable FileFilter fileFilter) throws MobileHarnessException {
        if (depth < 0) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_LIST_DIR_DEPTH_PARAM_ERROR, "The depth should not be less than 0.");
        }
        File dir = this.checkDir(dirPath);
        ArrayList<String> dirs = new ArrayList<String>();
        if (depth == 0) {
            if (recursively) {
                return dirs;
            }
            if (fileFilter == null || fileFilter.accept(dir)) {
                dirs.add(dirPath);
            }
        } else {
            for (File subFileOrDir : this.listFilesOrDirs(dir, null)) {
                String subFileOrDirPath = subFileOrDir.getAbsolutePath();
                if (recursively && (fileFilter == null || fileFilter.accept(subFileOrDir))) {
                    dirs.add(subFileOrDirPath);
                }
                if (!subFileOrDir.isDirectory()) continue;
                dirs.addAll(this.listDirs(subFileOrDirPath, depth - 1, recursively, fileFilter));
            }
        }
        return dirs;
    }

    public List<String> listFilePaths(String dirPath, boolean recursively) throws MobileHarnessException {
        return this.listFilePaths(dirPath, recursively, null);
    }

    public List<String> listFilePaths(String dirPath, boolean recursively, @Nullable FileFilter fileFilter) throws MobileHarnessException {
        ArrayList<String> filePaths = new ArrayList<String>();
        for (File file : this.listFilesOrDirs(dirPath, fileFilter)) {
            if (!file.isFile()) continue;
            filePaths.add(file.getAbsolutePath());
        }
        if (recursively) {
            for (File subDir : this.listDirs(dirPath)) {
                filePaths.addAll(this.listFilePaths(subDir.getAbsolutePath(), true, fileFilter));
            }
        }
        return filePaths;
    }

    public List<String> listFilePaths(Iterable<String> dirPaths, boolean recursively) throws MobileHarnessException {
        ArrayList<String> filePaths = new ArrayList<String>();
        for (String dirPath : dirPaths) {
            filePaths.addAll(this.listFilePaths(dirPath, recursively));
        }
        return filePaths;
    }

    public List<Path> listFilePaths(Path dir, boolean recursively) throws MobileHarnessException {
        return this.listFilePaths(dir, recursively, (Path dummy) -> true);
    }

    public List<Path> listFilePaths(final Path dir, final boolean recursively, final DirectoryStream.Filter<Path> filter) throws MobileHarnessException {
        this.checkDir(dir);
        try {
            final ArrayList<Path> files = new ArrayList<Path>();
            Files.walkFileTree(dir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path directory, BasicFileAttributes attrs) {
                    return recursively || directory.equals(dir) ? FileVisitResult.CONTINUE : FileVisitResult.SKIP_SUBTREE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (filter.accept(file)) {
                        files.add(file);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
            return files;
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_LIST_FILE_PATHS_ERROR, "Failed to list files of " + String.valueOf(dir), e);
        }
    }

    public List<File> listFiles(String dirPath, boolean recursively) throws MobileHarnessException {
        return this.listFiles(dirPath, recursively, null);
    }

    public List<File> listFiles(String dirPath, boolean recursively, @Nullable FileFilter fileFilter) throws MobileHarnessException {
        return this.listFiles(dirPath, recursively, fileFilter, null);
    }

    public List<File> listFiles(String dirPath, boolean recursively, @Nullable FileFilter fileFilter, @Nullable FileFilter dirFilter) throws MobileHarnessException {
        ArrayList<File> files = new ArrayList<File>();
        for (File file : this.listFilesOrDirs(dirPath, fileFilter)) {
            if (!file.isFile()) continue;
            files.add(file);
        }
        if (recursively) {
            for (File subDir : this.listDirs(dirPath, dirFilter)) {
                files.addAll(this.listFiles(subDir.getAbsolutePath(), true, fileFilter));
            }
        }
        return files;
    }

    public File[] listFilesOrDirs(String dirPath) throws MobileHarnessException {
        return this.listFilesOrDirs(dirPath, null);
    }

    public File[] listFilesOrDirs(String dirPath, @Nullable FileFilter filter) throws MobileHarnessException {
        return this.listFilesOrDirs(this.checkDir(dirPath), filter);
    }

    public List<Path> listFilesOrDirs(Path dir, DirectoryStream.Filter<Path> filter) throws MobileHarnessException {
        ArrayList<Path> arrayList;
        block9: {
            DirectoryStream<Path> subDirStream = Files.newDirectoryStream(dir, filter);
            try {
                ArrayList<Path> subDirs = new ArrayList<Path>();
                for (Path subDir : subDirStream) {
                    subDirs.add(subDir);
                }
                arrayList = subDirs;
                if (subDirStream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (subDirStream != null) {
                        try {
                            subDirStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_LIST_SUB_DIR_ERROR, "Failed to list sub directories of " + String.valueOf(dir), e);
                }
            }
            subDirStream.close();
        }
        return arrayList;
    }

    private File[] listFilesOrDirs(File dir, @Nullable FileFilter filter) throws MobileHarnessException {
        File[] files = dir.listFiles(filter);
        if (files == null) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_LIST_FILE_OR_DIRS_ERROR, String.format("Failed to list files or directories under %s", dir.getAbsolutePath()));
        }
        return files;
    }

    public List<String> listFileOrDirPaths(String dirPath) throws MobileHarnessException {
        return Arrays.stream(this.listFilesOrDirs(dirPath)).map(File::getAbsolutePath).collect(ImmutableList.toImmutableList());
    }

    public void mergeDir(Path srcDir, Path targetDir) throws MobileHarnessException, InterruptedException {
        this.mergeDir(srcDir.toAbsolutePath().toString(), targetDir.toAbsolutePath().toString());
    }

    public void mergeDir(String srcDirPath, String targetDirPath) throws MobileHarnessException, InterruptedException {
        this.checkDir(srcDirPath);
        this.prepareDir(targetDirPath, new FileAttribute[0]);
        try {
            this.cmdExecutor.exec(Command.of("rsync", "-a", "--no-group", "--no-owner", srcDirPath + "/", targetDirPath + "/"));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_MERGE_ERROR, String.format("Failed to merge dir %s into dir %s", srcDirPath, targetDirPath), e);
        }
        this.removeFileOrDir(srcDirPath);
    }

    public void moveFileOrDir(Path srcFileOrDirPath, Path desFileOrDirPath) throws MobileHarnessException, InterruptedException {
        this.moveFileOrDir(srcFileOrDirPath.toString(), desFileOrDirPath.toString());
    }

    public void moveFileOrDir(String srcFileOrDirPath, String desFileOrDirPath) throws MobileHarnessException, InterruptedException {
        try {
            this.cmdExecutor.exec(Command.of("mv", srcFileOrDirPath, desFileOrDirPath));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_MOVE_ERROR, String.format("Failed to move file/dir from %s to %s", srcFileOrDirPath, desFileOrDirPath), e);
        }
    }

    public BufferedWriter newBufferedWriter(Path filePath, OpenOption ... options) throws MobileHarnessException {
        this.prepareParentDir(filePath, new FileAttribute[0]);
        try {
            return Files.newBufferedWriter(filePath, options);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_NEW_BUFFERED_WRITER_ERROR, String.format("Failed to create new buffered writer on file %s", filePath), e);
        }
    }

    public void prepareDir(String dir, FileAttribute<?> ... attrs) throws MobileHarnessException {
        this.prepareDir(Paths.get(dir, new String[0]), attrs);
    }

    public void prepareDir(Path dir, FileAttribute<?> ... attrs) throws MobileHarnessException {
        try {
            Files.createDirectories(dir, attrs);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_CREATE_ERROR, "Failed to create directory " + String.valueOf(dir), e);
        }
    }

    public void prepareParentDir(String fileOrDir, FileAttribute<?> ... attrs) throws MobileHarnessException {
        String parentDir = new File(fileOrDir).getParent();
        if (parentDir != null) {
            this.prepareDir(parentDir, attrs);
        }
    }

    public void prepareParentDir(Path fileOrDir, FileAttribute<?> ... attrs) throws MobileHarnessException {
        this.prepareParentDir(fileOrDir.toString(), attrs);
    }

    public byte[] readBinaryFile(String filePath) throws MobileHarnessException {
        File file = this.checkFile(filePath);
        long maxFileSize = Integer.MAX_VALUE;
        long fileLength = 0L;
        try {
            fileLength = file.length();
        }
        catch (SecurityException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atWarning()).withCause(e)).log("Failed to get the length of file %s", filePath);
        }
        if (fileLength >= Integer.MAX_VALUE) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_TOO_LARGE_TO_READ, "Too large file " + filePath + ": " + fileLength + "Byte");
        }
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)fileLength);
        try (BufferedInputStream inputStream = new BufferedInputStream(new FileInputStream(file));){
            byte[] buffer = new byte[32768];
            int len = inputStream.read(buffer);
            while (len > 0) {
                outputStream.write(buffer, 0, len);
                len = inputStream.read(buffer);
            }
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_READ_BINARY_ERROR, "Failed to read the content of binary file " + filePath, e);
        }
        return outputStream.toByteArray();
    }

    public InputStream newInputStream(Path path) throws MobileHarnessException {
        try {
            return new BufferedInputStream(new FileInputStream(path.toFile()));
        }
        catch (FileNotFoundException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_NOT_FOUND, "File not found: " + String.valueOf(path), e);
        }
    }

    public String readFile(String filePath) throws MobileHarnessException {
        try {
            return LocalFileUtil.readFile(Files.newBufferedReader(this.checkFile(filePath).toPath(), StandardCharsets.UTF_8));
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_READ_STRING_ERROR, "Failed to read content of file " + filePath, e);
        }
    }

    private static String readFile(Reader reader) throws IOException {
        StringBuilder content = new StringBuilder();
        try (BufferedReader ignored = new BufferedReader(reader);){
            char[] buffer = new char[32768];
            int len = reader.read(buffer);
            while (len > 0) {
                content.append(buffer, 0, len);
                len = reader.read(buffer);
            }
        }
        return content.toString();
    }

    public String readFile(Path file) throws MobileHarnessException {
        try {
            return Files.readString(file);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_READ_STRING_ERROR, "Failed to read content of file " + String.valueOf(file), e);
        }
    }

    public String readFileHead(String filePath) throws MobileHarnessException, InterruptedException {
        this.checkFile(filePath);
        try {
            return this.cmdExecutor.run(Command.of("head", filePath).timeout(SLOW_CMD_TIMEOUT));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_READ_HEAD_ERROR, "Failed to read the head of file " + filePath, e);
        }
    }

    public String readFileTail(String filePath, int line) throws MobileHarnessException, InterruptedException {
        this.checkFile(filePath);
        try {
            return this.cmdExecutor.run(Command.of("tail", String.format("-%d", line), filePath).timeout(SLOW_CMD_TIMEOUT));
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_READ_TAIL_ERROR, "Failed to read the tail of file " + filePath, e);
        }
    }

    public Set<String> readLineSetFromFiles(Collection<String> filePaths) throws MobileHarnessException {
        HashSet<String> lines = new HashSet<String>();
        for (String filePath : filePaths) {
            try {
                String line;
                BufferedReader in = com.google.common.io.Files.newReader(new File(filePath), StandardCharsets.UTF_8);
                while ((line = in.readLine()) != null) {
                    lines.add(line);
                }
            }
            catch (IOException e) {
                throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_READ_LINES_FROM_FILE_SET, "Failed to read file(s) " + String.valueOf(filePaths), e);
            }
        }
        return lines;
    }

    public List<String> readLineListFromFile(String filePath) throws MobileHarnessException {
        ArrayList<String> lines = new ArrayList<String>();
        try {
            String line;
            BufferedReader in = com.google.common.io.Files.newReader(new File(filePath), StandardCharsets.UTF_8);
            while ((line = in.readLine()) != null) {
                lines.add(line);
            }
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_READ_LINES_FROM_FILE, "Failed to read file " + filePath, e);
        }
        return lines;
    }

    public Path readSymbolicLink(Path symbolicLink) throws MobileHarnessException {
        try {
            return Files.readSymbolicLink(symbolicLink);
        }
        catch (IOException | SecurityException | UnsupportedOperationException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_READ_SYMLINK_ERROR, "Failed to read the real path of symbolic link " + String.valueOf(symbolicLink), e);
        }
    }

    public void removeFileOrDir(Path fileOrDirPath) throws MobileHarnessException, InterruptedException {
        this.removeFileOrDir(fileOrDirPath.toString());
    }

    public void removeFileOrDir(String fileOrDirPath) throws MobileHarnessException, InterruptedException {
        try {
            this.cmdExecutor.exec(Command.of("rm", "-rf", fileOrDirPath).timeout(Timeout.fixed(SLOW_CMD_TIMEOUT)));
        }
        catch (MobileHarnessException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_OR_DIR_REMOVE_ERROR, "Failed to remove file/dir " + fileOrDirPath, e);
        }
    }

    public void removeFilesOrDirs(String dirPath, @Nullable String fileOrDirNameRegExp) throws MobileHarnessException, InterruptedException {
        File dir = this.checkDir(dirPath);
        Pattern pattern = fileOrDirNameRegExp == null ? null : Pattern.compile(fileOrDirNameRegExp);
        for (File fileOrDir : this.listFilesOrDirs(dir, null)) {
            if (pattern != null && !pattern.matcher(fileOrDir.getName()).matches()) continue;
            this.removeFileOrDir(fileOrDir.getAbsolutePath());
        }
    }

    public void removeFilesOrDirs(String dirPath) throws MobileHarnessException, InterruptedException {
        this.removeFilesOrDirs(dirPath, null);
    }

    public void removeFilesOrDirs(Path dir) throws MobileHarnessException, InterruptedException {
        this.removeFilesOrDirs(dir.toString(), null);
    }

    public void resetFile(String filePath) throws MobileHarnessException {
        try {
            this.writeToFile(filePath, "", false);
        }
        catch (MobileHarnessException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_RESET_ERROR, "Failed to clean up the content of file " + filePath, e);
        }
    }

    public boolean touchFileOrDir(String fileOrDirPath, boolean ifCreateNewFile) throws MobileHarnessException {
        File fileOrDir = this.getFileOrDir(fileOrDirPath);
        if (!fileOrDir.exists()) {
            if (ifCreateNewFile) {
                String parentDir = fileOrDir.getParent();
                if (parentDir != null) {
                    this.prepareDir(parentDir, new FileAttribute[0]);
                }
                try {
                    return fileOrDir.createNewFile();
                }
                catch (IOException e) {
                    throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_CREATE_NEW_ERROR, "Can not create empty file " + fileOrDirPath, e);
                }
            }
            return false;
        }
        return fileOrDir.setLastModified(System.currentTimeMillis());
    }

    public boolean touchFileOrDir(Path fileOrDir, boolean ifCreateNewFile) throws MobileHarnessException {
        return this.touchFileOrDir(fileOrDir.toString(), ifCreateNewFile);
    }

    public void setFilePermission(String filePath, String permission) throws MobileHarnessException {
        this.setFilePermission(Paths.get(filePath, new String[0]), permission);
    }

    public void setFilePermission(Path file, String permission) throws MobileHarnessException {
        try {
            Files.setPosixFilePermissions(file, PosixFilePermissions.fromString(permission));
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_SET_PERMISSION_ERROR, String.format("Failed to set permission %s to file %s", permission, file), e);
        }
    }

    public void writeToFile(String filePath, String content) throws MobileHarnessException {
        this.writeToFile(filePath, content, false);
    }

    public void writeToFile(String filePath, String content, boolean append) throws MobileHarnessException {
        this.prepareParentDir(filePath, new FileAttribute[0]);
        try {
            OpenOption[] openOptionArray;
            Path path = Path.of(filePath, new String[0]);
            if (append) {
                OpenOption[] openOptionArray2 = new StandardOpenOption[2];
                openOptionArray2[0] = StandardOpenOption.CREATE;
                openOptionArray = openOptionArray2;
                openOptionArray2[1] = StandardOpenOption.APPEND;
            } else {
                StandardOpenOption[] standardOpenOptionArray = new StandardOpenOption[3];
                standardOpenOptionArray[0] = StandardOpenOption.CREATE;
                standardOpenOptionArray[1] = StandardOpenOption.WRITE;
                openOptionArray = standardOpenOptionArray;
                standardOpenOptionArray[2] = StandardOpenOption.TRUNCATE_EXISTING;
            }
            try (BufferedWriter out = Files.newBufferedWriter(path, StandardCharsets.UTF_8, openOptionArray);){
                out.write(content);
                out.flush();
            }
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_WRITE_STRING_ERROR, "Failed to write content to file " + filePath, e);
        }
    }

    public long writeToFile(Path filePath, byte[] input, boolean append) throws MobileHarnessException {
        this.prepareParentDir(filePath, new FileAttribute[0]);
        try {
            OpenOption[] openOptionArray;
            if (append) {
                OpenOption[] openOptionArray2 = new StandardOpenOption[2];
                openOptionArray2[0] = StandardOpenOption.CREATE;
                openOptionArray = openOptionArray2;
                openOptionArray2[1] = StandardOpenOption.APPEND;
            } else {
                StandardOpenOption[] standardOpenOptionArray = new StandardOpenOption[3];
                standardOpenOptionArray[0] = StandardOpenOption.CREATE;
                standardOpenOptionArray[1] = StandardOpenOption.WRITE;
                openOptionArray = standardOpenOptionArray;
                standardOpenOptionArray[2] = StandardOpenOption.TRUNCATE_EXISTING;
            }
            Files.write(filePath, input, openOptionArray);
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_WRITE_BYTE_ERROR, "Failed to write content to file " + String.valueOf(filePath), e);
        }
        return input.length;
    }

    public long writeToFile(String filePath, byte[] input, boolean append) throws MobileHarnessException {
        return this.writeToFile(Path.of(filePath, new String[0]), input, append);
    }

    @CanIgnoreReturnValue
    public long writeToFile(Path filePath, byte[] input) throws MobileHarnessException {
        return this.writeToFile(filePath, input, false);
    }

    public long writeToFile(String filePath, byte[] input) throws MobileHarnessException {
        return this.writeToFile(Path.of(filePath, new String[0]), input);
    }

    public long writeToFile(String filePath, InputStream input) throws MobileHarnessException {
        long l;
        Preconditions.checkNotNull(input);
        this.prepareParentDir(filePath, new FileAttribute[0]);
        long fileSize = 0L;
        BufferedOutputStream output = new BufferedOutputStream(new FileOutputStream(filePath));
        try {
            byte[] buf = new byte[32768];
            int len = input.read(buf);
            while (len > 0) {
                output.write(buf, 0, len);
                fileSize += (long)len;
                len = input.read(buf);
            }
            output.flush();
            l = fileSize;
        }
        catch (Throwable throwable) {
            try {
                try {
                    output.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | SecurityException e) {
                throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_WRITE_STREAM_ERROR, "Failed to write stream to file " + filePath, e);
            }
        }
        output.close();
        return l;
    }

    public void extractTarGz(Path tarGzPath, Path targetDirPath) throws MobileHarnessException, InterruptedException {
        this.extractTarGz(tarGzPath.toString(), targetDirPath.toString());
    }

    public void extractTarGz(String tarGzPath, String targetDirPath) throws MobileHarnessException, InterruptedException {
        try {
            Command command = Command.of(String.format("tar -zxvf %s -C %s", tarGzPath, targetDirPath).split(" "));
            this.cmdExecutor.exec(command);
        }
        catch (CommandException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_TAR_EXTRACT_ERROR, String.format("Failed to extract file %s to directory %s", tarGzPath, targetDirPath), e);
        }
    }

    public String unzipFile(String zipFilePath, String targetDirPath) throws MobileHarnessException, InterruptedException {
        return this.unzipFile(zipFilePath, targetDirPath, (Duration)null);
    }

    public String unzipFile(Path zipFile, Path targetDir) throws MobileHarnessException, InterruptedException {
        return this.unzipFile(zipFile.toString(), targetDir.toString());
    }

    public String unzipFile(String zipFilePath, String targetDirPath, @Nullable Duration timeout) throws MobileHarnessException, InterruptedException {
        try {
            this.prepareDir(targetDirPath, new FileAttribute[0]);
            Command command = Command.of("unzip", "-o", zipFilePath).workDir(targetDirPath);
            if (timeout != null) {
                command = command.timeout(Timeout.fixed(timeout));
            }
            return this.cmdExecutor.run(command);
        }
        catch (MobileHarnessException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_UNZIP_ERROR, String.format("Failed to unzip file %s to dir %s", zipFilePath, targetDirPath), e);
        }
    }

    public String unzipFile(String zipFilePath, String fileNameToUnzip, String targetDirPath) throws MobileHarnessException, InterruptedException {
        return this.unzipFiles(zipFilePath, ImmutableList.of(fileNameToUnzip), targetDirPath);
    }

    public String unzipFiles(String zipFilePath, List<String> fileNamesToUnzip, String targetDirPath) throws MobileHarnessException, InterruptedException {
        try {
            this.prepareDir(targetDirPath, new FileAttribute[0]);
            ImmutableCollection cmd = ((ImmutableList.Builder)((ImmutableList.Builder)((ImmutableList.Builder)((ImmutableList.Builder)new ImmutableList.Builder().add("unzip")).add("-o")).add(zipFilePath)).addAll(fileNamesToUnzip)).build();
            Command command = Command.of((List<String>)((Object)cmd)).workDir(targetDirPath);
            return this.cmdExecutor.run(command);
        }
        catch (MobileHarnessException e) {
            if (e.getErrorId() == BasicErrorId.COMMAND_EXEC_FAIL && e.getMessage().contains("filename not matched")) {
                throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_UNZIP_FILENAME_NOT_MATCHED, String.format("Failed to unzip file %s from %s because the filename %s is not matched", fileNamesToUnzip, zipFilePath, fileNamesToUnzip), e);
            }
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_UNZIP_PARTICULAR_FILES_ERROR, String.format("Failed to unzip files %s from %s to dir %s", fileNamesToUnzip, zipFilePath, targetDirPath), e);
        }
    }

    public BufferedWriter writeToFileWithBufferedWriter(@Nullable BufferedWriter bufferedWriter, String filePath, String data, boolean append) throws MobileHarnessException {
        if (bufferedWriter == null) {
            try {
                bufferedWriter = this.createBufferedWriter(filePath, append);
            }
            catch (IOException e) {
                throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_WRITE_STREAM_CREATE_BUFFER_ERROR, "Failed to write content to file " + filePath, e);
            }
        }
        try {
            bufferedWriter.write(data);
            bufferedWriter.flush();
        }
        catch (IOException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_FILE_WRITE_STREAM_WITH_BUFFER_ERROR, "Failed to write content to file " + filePath, e);
        }
        return bufferedWriter;
    }

    public String zipDir(String sourceDirPath, String zipFilePath) throws MobileHarnessException, InterruptedException {
        return this.zipDir(sourceDirPath, zipFilePath, false, false, null, null, false);
    }

    public String zipDir(String sourceDirPath, String zipFilePath, boolean keepLocalSourceRootBaseName) throws MobileHarnessException, InterruptedException {
        return this.zipDir(sourceDirPath, zipFilePath, false, false, null, null, keepLocalSourceRootBaseName);
    }

    public String zipDir(String sourceDirPath, String zipFilePath, boolean sortFile, boolean storeOnly, @Nullable Integer compressionLevel, @Nullable Timeout timeout) throws MobileHarnessException, InterruptedException {
        return this.zipDir(sourceDirPath, zipFilePath, sortFile, storeOnly, compressionLevel, timeout, false);
    }

    public String zipDir(String sourceDirPath, String zipFilePath, boolean sortFile, boolean storeOnly, @Nullable Integer compressionLevel, @Nullable Timeout timeout, boolean keepLocalSourceRootBaseName) throws MobileHarnessException, InterruptedException {
        Path absSourceDirPath = Paths.get(sourceDirPath, new String[0]).toAbsolutePath();
        ImmutableCollection.Builder arguments = ImmutableList.builder().add(new String[]{"zip", "-X"});
        if (storeOnly) {
            ((ImmutableList.Builder)arguments).add("-0");
        }
        if (compressionLevel != null) {
            if (compressionLevel >= 1 && compressionLevel <= 9) {
                ((ImmutableList.Builder)arguments).add("-" + compressionLevel);
            } else {
                throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_ZIP_ERROR, String.format("Failed to zip dir %s into %s. The compressionLevel should between 1 to 9", sourceDirPath, zipFilePath));
            }
        }
        ((ImmutableList.Builder)arguments).add(new String[]{"-r", zipFilePath});
        if (sortFile) {
            List<String> files = this.listFilePaths(absSourceDirPath.toString(), true);
            if (files.size() < 1000) {
                ((ImmutableList.Builder)arguments).addAll((Iterable)files.stream().sorted().map(file -> keepLocalSourceRootBaseName ? PathUtil.join(PathUtil.basename(sourceDirPath), absSourceDirPath.relativize(Path.of(file, new String[0]).toAbsolutePath()).toString()) : absSourceDirPath.relativize(Path.of(file, new String[0]).toAbsolutePath()).toString()).collect(ImmutableList.toImmutableList()));
            } else {
                ((ImmutableList.Builder)arguments).add(keepLocalSourceRootBaseName ? PathUtil.basename(sourceDirPath) : ".");
            }
        } else {
            ((ImmutableList.Builder)arguments).add(keepLocalSourceRootBaseName ? PathUtil.basename(sourceDirPath) : ".");
        }
        Path workDir = absSourceDirPath;
        if (keepLocalSourceRootBaseName && absSourceDirPath.getParent() != null) {
            workDir = absSourceDirPath.getParent();
        }
        Command command = Command.of((List<String>)((Object)((ImmutableList.Builder)arguments).build())).workDir(workDir);
        if (timeout != null) {
            command = command.timeout(timeout);
        }
        try {
            return this.cmdExecutor.run(command);
        }
        catch (MobileHarnessException e) {
            throw new MobileHarnessException(BasicErrorId.LOCAL_DIR_ZIP_ERROR, String.format("Failed to zip dir %s into %s", sourceDirPath, zipFilePath), e);
        }
    }

    private BufferedWriter createBufferedWriter(String filePath, boolean append) throws MobileHarnessException, IOException {
        OpenOption[] openOptionArray;
        this.prepareParentDir(filePath, new FileAttribute[0]);
        Path path = Paths.get(filePath, new String[0]);
        if (append) {
            OpenOption[] openOptionArray2 = new StandardOpenOption[2];
            openOptionArray2[0] = StandardOpenOption.CREATE;
            openOptionArray = openOptionArray2;
            openOptionArray2[1] = StandardOpenOption.APPEND;
        } else {
            StandardOpenOption[] standardOpenOptionArray = new StandardOpenOption[3];
            standardOpenOptionArray[0] = StandardOpenOption.CREATE;
            standardOpenOptionArray[1] = StandardOpenOption.WRITE;
            openOptionArray = standardOpenOptionArray;
            standardOpenOptionArray[2] = StandardOpenOption.TRUNCATE_EXISTING;
        }
        return Files.newBufferedWriter(path, StandardCharsets.UTF_8, openOptionArray);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean areAllFilesFullAccessible(Path fileOrDir) {
        if (!this.isFullyAccessible(fileOrDir)) {
            return false;
        }
        if (!Files.isDirectory(fileOrDir, new LinkOption[0])) return true;
        try (DirectoryStream<Path> subFileStream = Files.newDirectoryStream(fileOrDir);){
            for (Path sub : subFileStream) {
                if (this.isFullyAccessible(sub)) continue;
                boolean bl = false;
                return bl;
            }
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atWarning()).withCause(e)).log("Failed to check the accessibility for file or path %s", fileOrDir);
            return false;
        }
    }

    private boolean isFullyAccessible(Path path) {
        try {
            String filePermissionString = this.getFilePermissionString(path);
            if (!filePermissionString.equals("rwxrwxrwx")) {
                ((FluentLogger.Api)logger.atFine()).log("Checked fully accessibility of file %s. Permissions: %s", (Object)path, (Object)filePermissionString);
                return false;
            }
            return true;
        }
        catch (MobileHarnessException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atWarning()).withCause(e)).log("Failed to get file permission for path %s", path);
            return false;
        }
    }
}

