/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.mobileharness.shared.util.command.backend;

import com.google.common.base.Preconditions;
import com.google.common.base.Verify;
import com.google.common.math.LongMath;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.Monitor;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.Arrays;

final class CapturingOutputStream
extends OutputStream {
    private static final int MAX_BUFFER_SIZE = 0x7FFFFFF7;
    private final Monitor monitor = new Monitor();
    private volatile byte[] data = new byte[32];
    private volatile int size = 0;
    private volatile boolean closed;

    CapturingOutputStream() {
    }

    InputStream openInputStream() {
        return new CapturedInputStream();
    }

    byte[] toByteArray() {
        return Arrays.copyOf(this.data, this.size);
    }

    String toString(Charset charset) {
        return new String(this.data, 0, this.size, charset);
    }

    @Override
    public void write(int b) throws IOException {
        this.ensureCapacityToWrite(1);
        this.data[this.size] = (byte)b;
        ++this.size;
        this.signalBytesWrittenOrStreamClosed();
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        Preconditions.checkPositionIndexes(off, off + len, b.length);
        this.ensureCapacityToWrite(len);
        System.arraycopy(b, off, this.data, this.size, len);
        this.size += len;
        this.signalBytesWrittenOrStreamClosed();
    }

    private void ensureCapacityToWrite(int numBytesToWrite) throws IOException {
        long minCapacity = LongMath.checkedAdd(this.size, numBytesToWrite);
        if (minCapacity > (long)this.data.length) {
            long newCapacity = Math.min(Math.max((long)this.data.length * 2L, minCapacity), 0x7FFFFFF7L);
            if (newCapacity < minCapacity) {
                throw new IOException("Cannot allocate enough memory to capture all output");
            }
            this.data = Arrays.copyOf(this.data, Ints.checkedCast(newCapacity));
        }
    }

    @Override
    public void close() throws IOException {
        this.closed = true;
        this.signalBytesWrittenOrStreamClosed();
    }

    private void signalBytesWrittenOrStreamClosed() {
        this.monitor.enter();
        this.monitor.leave();
    }

    private final class CapturedInputStream
    extends InputStream {
        private int position = 0;
        private int mark = 0;
        private final Monitor.Guard bytesAvailableOrClosed;

        private CapturedInputStream() {
            this.bytesAvailableOrClosed = new Monitor.Guard(CapturingOutputStream.this.monitor){

                @Override
                public boolean isSatisfied() {
                    return CapturedInputStream.this.position < CapturingOutputStream.this.size || CapturingOutputStream.this.closed;
                }
            };
        }

        private int waitForBytes() throws IOException {
            if (!this.bytesAvailableOrClosed.isSatisfied()) {
                try {
                    CapturingOutputStream.this.monitor.enterWhen(this.bytesAvailableOrClosed);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IOException(e);
                }
                CapturingOutputStream.this.monitor.leave();
            }
            return CapturingOutputStream.this.size - this.position;
        }

        @Override
        public int read() throws IOException {
            this.verifyPosition();
            return this.waitForBytes() == 0 ? -1 : CapturingOutputStream.this.data[this.position++];
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            this.verifyPosition();
            Preconditions.checkPositionIndexes(off, off + len, b.length);
            if (len == 0) {
                return 0;
            }
            int numBytesAvailable = this.waitForBytes();
            if (numBytesAvailable == 0) {
                return -1;
            }
            int numBytesRead = Math.min(numBytesAvailable, len);
            System.arraycopy(CapturingOutputStream.this.data, this.position, b, off, numBytesRead);
            this.position += numBytesRead;
            return numBytesRead;
        }

        @Override
        public long skip(long n) throws IOException {
            this.verifyPosition();
            if (n <= 0L) {
                return 0L;
            }
            int numBytesSkipped = (int)Math.min(n, (long)this.waitForBytes());
            this.position += numBytesSkipped;
            return numBytesSkipped;
        }

        @Override
        public int available() throws IOException {
            this.verifyPosition();
            return CapturingOutputStream.this.size - this.position;
        }

        @Override
        public void mark(int readlimit) {
            this.mark = this.position;
        }

        @Override
        public void reset() throws IOException {
            this.position = this.mark;
        }

        @Override
        public boolean markSupported() {
            return true;
        }

        private void verifyPosition() {
            Verify.verify(this.position >= 0);
            Verify.verify(this.position <= CapturingOutputStream.this.size);
        }
    }
}

