/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.mobileharness.shared.util.port;

import com.google.common.base.Preconditions;
import com.google.common.flogger.FluentLogger;
import com.google.devtools.mobileharness.shared.util.port.UnableToFindPortException;
import com.google.devtools.mobileharness.shared.util.port.UnexpectedPortStateException;
import java.io.IOException;
import java.net.ConnectException;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.time.Duration;
import java.util.Random;

public class PortProber {
    private static final int RANDOM_PORT_BASE = 32768;
    private static final int RANDOM_PORT_RANGE = 27233;
    private static final int INVALID_PORT = -1;
    private static final Random random = new SecureRandom();
    private static final ByteBuffer PORT_SERVER_REQUEST_BUFFER = ByteBuffer.allocateDirect(512);
    private static final String PORT_SERVER_PORT_ENV_VAR = "PORTSERVER_PORT";
    private static final Duration PORT_FREE_TIMEOUT = Duration.ofMillis(10L);
    private static final int SOCKET_TIMEOUT_IN_MS = 5000;
    private static final int PORT_SERVER_PORT;
    private static final String PID_STRING;
    private static final FluentLogger logger;

    public static void checkLocalPortIsFree(int port) throws IOException {
        if (PortProber.isPortOpen("localhost", port)) {
            throw new UnexpectedPortStateException("the local port is busy");
        }
    }

    private static void checkLocalPortBecomesFree(int port, Duration timeout) throws IOException, InterruptedException {
        if (PortProber.isPortOpen("localhost", port)) {
            Thread.sleep(timeout.toMillis());
            PortProber.checkLocalPortIsFree(port);
        }
    }

    public static void checkLocalPortIsBusy(int port) throws IOException {
        if (!PortProber.isPortOpen("localhost", port)) {
            throw new UnexpectedPortStateException("the local port is free");
        }
    }

    public static boolean isPortOpen(String host, int port) throws IOException {
        try {
            Socket socket = new Socket(host, port);
            socket.close();
            return true;
        }
        catch (ConnectException e) {
            PortProber.assertStartsWith("Connection refused", e.getMessage());
            return false;
        }
        catch (SocketException e) {
            PortProber.assertStartsWith("Connection reset by peer", e.getMessage());
            return false;
        }
    }

    private static void assertStartsWith(String expectedPrefix, String actual) {
        if (actual != null && actual.startsWith(expectedPrefix)) {
            return;
        }
        throw new AssertionError((Object)("[" + actual + "] didn't start with [" + expectedPrefix + "]"));
    }

    public static int pickUnusedPort() throws IOException, InterruptedException {
        return PortProber.pickUnusedPort(32768, 27233);
    }

    public static int pickUnusedPort(int startPort, int portRangeLength) {
        if (PORT_SERVER_PORT != 0) {
            int port = PortProber.pickPortFromPortServer();
            if (port > 0 && port < 65536) {
                return port;
            }
            throw new UnableToFindPortException("Couldn't get port from portserver.");
        }
        return PortProber.scanForUnusedPort(startPort, portRangeLength);
    }

    private static int scanForUnusedPort(int minPort, int portRangeLength) {
        int currentPort;
        Preconditions.checkArgument(portRangeLength > 0, "Must specify a portRangeLength of 1 or greater");
        int maxPort = minPort + portRangeLength - 1;
        int startPort = PortProber.getRandomStartingPort(minPort, portRangeLength);
        int foundPort = -1;
        SocketType portType = SocketType.TCP;
        for (currentPort = startPort; foundPort == -1 && currentPort <= maxPort; ++currentPort) {
            foundPort = PortProber.checkPortAt(currentPort, portType);
            portType = PortProber.togglePortType(portType);
        }
        for (currentPort = minPort; foundPort == -1 && currentPort < startPort; ++currentPort) {
            foundPort = PortProber.checkPortAt(currentPort, portType);
            portType = PortProber.togglePortType(portType);
        }
        if (foundPort == -1) {
            ((FluentLogger.Api)logger.atInfo()).log("No port found within requested range.  Picking system-provided port.");
            foundPort = PortProber.checkPortAt(0, portType);
        }
        return foundPort;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static int getRandomStartingPort(int minPort, int range) {
        Random random = PortProber.random;
        synchronized (random) {
            return minPort + PortProber.random.nextInt(range);
        }
    }

    private static SocketType togglePortType(SocketType portType) {
        return portType == SocketType.TCP ? SocketType.UDP : SocketType.TCP;
    }

    private static int checkPortAt(int portToTry, SocketType portType) {
        int candidatePort = PortProber.checkPortForListen(portToTry, portType);
        if (candidatePort != -1 && (candidatePort = PortProber.checkPortForListen(candidatePort, PortProber.togglePortType(portType))) != -1) {
            try {
                PortProber.checkLocalPortBecomesFree(candidatePort, PORT_FREE_TIMEOUT);
            }
            catch (IOException | InterruptedException e) {
                e.printStackTrace();
                candidatePort = -1;
            }
        }
        return candidatePort;
    }

    private static synchronized int pickPortFromPortServer() {
        long start = System.currentTimeMillis();
        try {
            SocketChannel channel = SocketChannel.open();
            InetSocketAddress address = new InetSocketAddress("localhost", PORT_SERVER_PORT);
            Socket socket = channel.socket();
            socket.setSoTimeout(5000);
            socket.connect(address, 5000);
            return PortProber.pickPortFromPortServerSocket(channel);
        }
        catch (UnknownHostException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atSevere()).withCause(e)).log("Unable to resolve PortServer at localhost:%d\n", PORT_SERVER_PORT);
            return 0;
        }
        catch (IOException e) {
            long duration = System.currentTimeMillis() - start;
            ((FluentLogger.Api)((FluentLogger.Api)logger.atSevere()).withCause(e)).log("Unable to connect to PortServer (waited %dms):\n", duration);
            return 0;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static synchronized int pickPortFromPortServerSocket(SocketChannel portServerSocketChannel) {
        Preconditions.checkNotNull(portServerSocketChannel);
        String portServerResponse = "";
        try {
            try {
                PORT_SERVER_REQUEST_BUFFER.clear();
                PORT_SERVER_REQUEST_BUFFER.put(PID_STRING.getBytes(StandardCharsets.UTF_8));
                PORT_SERVER_REQUEST_BUFFER.flip();
                if (portServerSocketChannel.write(PORT_SERVER_REQUEST_BUFFER) <= 0) {
                    ((FluentLogger.Api)logger.atSevere()).log("Unexpected failure writing to PortServer buffer.");
                    int n = 0;
                    return n;
                }
                PORT_SERVER_REQUEST_BUFFER.clear();
                if (portServerSocketChannel.read(PORT_SERVER_REQUEST_BUFFER) <= 0) {
                    ((FluentLogger.Api)logger.atSevere()).log("Unexpected failure reading from PortServer buffer.");
                    int n = 0;
                    return n;
                }
                PORT_SERVER_REQUEST_BUFFER.flip();
                byte[] bytes = new byte[PORT_SERVER_REQUEST_BUFFER.limit() - PORT_SERVER_REQUEST_BUFFER.position()];
                PORT_SERVER_REQUEST_BUFFER.get(bytes);
                portServerResponse = new String(bytes, StandardCharsets.UTF_8).trim();
                int port = Integer.parseInt(portServerResponse);
                if (port > 0 && port < 65536) {
                    int n = port;
                    return n;
                }
                ((FluentLogger.Api)logger.atWarning()).log("Got invalid port number from PortServer: %d", port);
                int n = 0;
                return n;
            }
            finally {
                portServerSocketChannel.socket().close();
            }
        }
        catch (IOException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atSevere()).withCause(e)).log("Exception while communicating with PortServer.");
            return 0;
        }
        catch (NumberFormatException e) {
            ((FluentLogger.Api)((FluentLogger.Api)logger.atSevere()).withCause(e)).log("Got unexpected NumberFormatException while parsing PortServer response: %s", portServerResponse);
            return 0;
        }
    }

    private static int checkPortForListen(int port, SocketType type) {
        if (port < 0) {
            throw new IllegalArgumentException("Invalid port (<0) specified");
        }
        switch (type.ordinal()) {
            case 1: {
                try {
                    ServerSocket socket = new ServerSocket(port);
                    int realPort = socket.getLocalPort();
                    socket.close();
                    return realPort;
                }
                catch (IOException ioe) {
                    return -1;
                }
            }
            case 0: {
                try {
                    DatagramSocket udpSocket = new DatagramSocket(port);
                    int realPort = udpSocket.getLocalPort();
                    udpSocket.close();
                    return realPort;
                }
                catch (SocketException se) {
                    return -1;
                }
            }
        }
        throw new IllegalArgumentException("Invalid type specified: " + String.valueOf((Object)type));
    }

    private PortProber() {
    }

    static {
        logger = FluentLogger.forEnclosingClass();
        int portServerPortSetting = 0;
        String pidStringSetting = null;
        String portServerPortString = System.getenv(PORT_SERVER_PORT_ENV_VAR);
        if (null != portServerPortString) {
            try {
                int port = Integer.parseInt(portServerPortString);
                if (port > 0 && port < 65536) {
                    pidStringSetting = ProcessHandle.current().pid() + "\n";
                    portServerPortSetting = port;
                }
            }
            catch (NumberFormatException e) {
                ((FluentLogger.Api)((FluentLogger.Api)logger.atSevere()).withCause(e)).log("Could not parse port from portServerPort: %s", portServerPortString);
            }
        } else {
            ((FluentLogger.Api)logger.atInfo()).log("%s not set in process environment, so not using PortServer.", PORT_SERVER_PORT_ENV_VAR);
        }
        PORT_SERVER_PORT = portServerPortSetting;
        PID_STRING = pidStringSetting;
    }

    static enum SocketType {
        UDP,
        TCP;

    }
}

